# -*- coding: utf-8 -*-

"""
Created 04-Jan-2026
See VersionInfo below in __main__ for current update

This script gets the total editing time of the active DGN file.
It uses the EC API to query TotalEditingTime
defined in ..\\MicroStation\\ECSchemas\\DGN\\DgnFileSchema.01.00.ecschema.xml.
"""

from MSPyBentley import *
from MSPyBentleyGeom import *
from MSPyECObjects import *
from MSPyDgnPlatform import *
from MSPyDgnView import *
from MSPyMstnPlatform import *

#from version_info import VersionInfo
from la_solutions.version_info import VersionInfo
from MSPyMstnPlatform import MessageCenter
 
def GetStoredSchemaList():
    """
    Get a list of schemas stored in the current DGN file.

    :returns: List of schemas stored in the current DGN file.
    :rtype: list
    """
    #list to collect schema names
    schemas = list()
    #Get dgnEC manager instance
    ec_manager = DgnECManager.GetManager()
    # Get the current DGN model
    #ACTIVEMODEL = ISessionMgr.ActiveDgnModelRef
    #dgnModel = ACTIVEMODEL.GetDgnModel()

    dgn_file = ISessionMgr.GetActiveDgnFile()

    #discover schemas
    infos = SchemaInfoArray()
    ec_manager.DiscoverSchemas(infos, dgn_file, 
                                ECSchemaPersistence.eECSCHEMAPERSISTENCE_All,
                                DgnECHostType.eFile)  
    print(f"SchemaInfoArray length={len(infos)}")
    info_container = SchemaInfoArray()
    for info in infos:
        info_container.append(info)
        
        if len(info_container) <= 0:
            continue
        
    #iterate container
    for info in info_container:
        #locate schema
        schema = ec_manager.LocateSchemaInDgnFile(info, SchemaMatchType.eSCHEMAMATCHTYPE_Exact)
        if schemas is None:
            continue  
        #get schema name                  
        #schemaName = ecSchema.GetFullSchemaName()
        #add to list
        schemas.append(schema)

    return schemas
 
class ECInstanceSerializer:
    '''
    This class has responsibility to serialize property names and values in dictionary.
    
    Copied from ECDump example.
    ''' 
    def __init__(self, instance):
        self._instance = instance

    def SerializePropValues (dgnECInstance, collection, ecPropNamValDict):
        """
        Serialize property values from a collection into a dictionary.

        :param dgnECInstance: The EC instance containing the property values.
        :type dgnECInstance: ECInstance
        :param collection: The collection of property values to serialize.
        :type collection: list
        :param ecPropNamValDict: The dictionary to store serialized property names and values.
        :type ecPropNamValDict: dict
        """
        #iterate through propety value collection
        for propertyValue in collection:
            #get accessor
            accessor = propertyValue.GetValueAccessor()
            #get managed access string
            accessString = accessor.GetManagedAccessString ()
            #get value, this is raw value
            value = propertyValue.GetValue()
            typeAdaptedValue = WString ()
            strAccessString = str(accessString)
            #If value is primitive, try getting typeAdapted value, else get value from instance directly
            #if value.IsStruct() == False and value.IsArray() == False:
            if not value.IsStruct() and not value.IsArray():
                if BentleyStatus.eSUCCESS == dgnECInstance.GetValueAsString(typeAdaptedValue, strAccessString, False, 0):
                    ecPropNamValDict[str(accessString)] = str(typeAdaptedValue)
                else:
                    dgnECInstance.GetValue(value, strAccessString)
                    ecPropNamValDict[str(accessString)] = str(value.ToString())
            #handle child values
            if (propertyValue.HasChildValues ()):
                ECInstanceSerializer.SerializePropValues (dgnECInstance, propertyValue.GetChildValues(), ecPropNamValDict)

    def SerializeInstance (instance, ecPropNamValDict):
        """
        Serialize an EC instance into a dictionary of property names and values.

        :param instance: The EC instance to serialize.
        :type instance: ECInstance
        :param ecPropNamValDict: The dictionary to store serialized property names and values.
        :type ecPropNamValDict: dict
        :returns: Dictionary of property names and values.
        :rtype: dict
        """
        #ECSchema
        ecPropNamValDict['Instance schema'] = str(instance.GetClass().GetSchema().GetName())
        #ECClass
        ecPropNamValDict['Instance class'] = str(instance.GetClass().GetName())
        #InstanceID
        ecPropNamValDict['Instance ID'] = str(instance.GetInstanceId())
        #get property value collection
        collection = ECValuesCollection.Create (instance)
        #serialize values
        ECInstanceSerializer.SerializePropValues(instance, collection, ecPropNamValDict)
    
    def Serialize(self):
        """
        Returns a dictionary of property names and values.

        :returns: Dictionary of property names and values.
        :rtype: dict
        """
        ecPropNamValDict = dict()
        ECInstanceSerializer.SerializeInstance(self._instance, ecPropNamValDict)
        return ecPropNamValDict

def extract_ec_data_from_dgnfile(dgn_file=None):
    """
    Extracts all EC data from the given DGN file.
    If no dgnFile is provided, uses the active DGN file.
    """
    # Get the DGN file and model
    if dgn_file is None:
        dgn_file = ISessionMgr.GetActiveDgnFile()
    if dgn_file is None:
        print("No active DGN file found.")
        return

    mgr = DgnECManager.GetManager()
    # Prepare EC query and scope:
    # query = what to look for
    # scope = where to look
    # We want to look for a specific EC schema and EC class...
    query = ECQuery.CreateQuery(eECQUERY_PROCESS_Empty)
    _EC_DGN_FILE_SCHEMA = 'DgnFileSchema'
    _EC_DGN_FILE_PROPERTIES_CLASS = 'DgnFileProperties'
    query.AddSearchClass(_EC_DGN_FILE_SCHEMA, _EC_DGN_FILE_PROPERTIES_CLASS)
    scope = FindInstancesScope.CreateScope(dgn_file, FindInstancesScopeOption(DgnECHostType.eFile))

    # Find EC instances
    instance_collection, n_instances = mgr.FindInstances(scope, query)
    print(f"instance_collection N instances={n_instances} type={type(instance_collection)}")
    # Serialize EC data
    ec_data_list = []
    #inst: MSPyECObjects.DgnECInstance
    for inst in instance_collection:
        print(f"inst type={type(inst)}")
        serializer = ECInstanceSerializer(inst)
        ec_data = serializer.Serialize()
        ec_data_list.append(ec_data)
 
    return ec_data_list

def write_csv_file(data: dict, file_name: str, field_names: list[str]=None)->int:
    import csv
    """
    Write a dictionary as a CSV file having the specified name.
    
    Returns: count of lines written.
    """
    n_data = 0
    print(f"write_csv_file '{file_name}' data type={type(data)}")
    with open(file_name, 'w', newline='') as csvfile:
        writer = csv.writer(csvfile)
        writer.writerow(fieldnames)
        for row in data.keys():            
            writer.writerow([row, data[row]]) 
            n_data += 1 
    return n_data

def ComposeTempFileName(suggestion)->str:
    import tempfile
    from pathlib import Path, WindowsPath
    temp = WindowsPath(tempfile.gettempdir())
    tempfile = os.path.join(temp, suggestion)  
    return tempfile
  
if __name__ == "__main__":  # check if this script is being run directly (not imported as a module)
    vinfo = VersionInfo("Total Edit Time", 26, 1, 7, "Show total editing time of a DGN file")
    MessageCenter.ShowInfoMessage(vinfo.brief, vinfo.verbose, False)

    schemas = GetStoredSchemaList()
    print("EC schemas in active DGN file")
    for schema in schemas:
        print(f"schema '{schema.GetName()}' ") 
        
    #  This list contains key/value pairs of EC data found in the active DGN file
    ec_data_list = extract_ec_data_from_dgnfile()
    for data in ec_data_list: 
        for o in data:
            #print(type(o), o)
            print(f"prop '{o}' value '{data[o]}'")
    
    print("Get specified EC properties...")
    CREATE_DATE_PROPERTY = 'CreateDate'
    create_date = data[CREATE_DATE_PROPERTY]
    print(f"{CREATE_DATE_PROPERTY}='{create_date}'") 
    TOTAL_EDIT_TIME_PROPERTY =  'TotalEditingTime'       
    total_edit_time = data[TOTAL_EDIT_TIME_PROPERTY]
    print(f"{TOTAL_EDIT_TIME_PROPERTY}='{total_edit_time}'")            
        
    tempfile = ComposeTempFileName('ec_data.csv')
    fieldnames = ['Property', 'Value']
    size = write_csv_file (data, tempfile, fieldnames)  
    print(f"Wrote {size} values to file '{tempfile}'")  
        
        