# -*- coding: utf-8 -*-

'''
Created 15-Jan-2025
See VersionInfo in __main__ for current update
'''
from MSPyBentley import *
from MSPyDgnPlatform import *
from MSPyMstnPlatform import *
from la_solutions.version_info import VersionInfo
import numpy as np
 
def TextSourceIdentity (eh: ElementHandle)->(str, int):
    # A text element may be found stand-alone or embedded in a text node, cell or text table.
    # Some applications may find it useful to know the source of a text element.
    # This function returns a tuple containing the source description and its Element ID.
    elementId = eh.GetElementId()
    source = DescribeTextSource(eh.GetHandler())
    return (source, elementId)

def DescribeTextSource (handler: Handler)->str:
    #   Where text is found
    TEXT_SOURCE_PRIMITIVE = "text element"
    TEXT_SOURCE_CELL      = "cell"
    TEXT_SOURCE_NODE      = "text node"
    TEXT_SOURCE_TABLE     = "text table"
    TEXT_SOURCE_UNKNOWN   = "unknown"
    # A text element may be found stand-alone or embedded in a text node, cell or text table.
    # Some applications may find it useful to know the source of a text element.
    match handler.__class__.__qualname__:
        case TextElemHandler.__qualname__:
            return TEXT_SOURCE_PRIMITIVE
        case TextNodeHandler.__qualname__:
            return TEXT_SOURCE_NODE
        case TextTableHandler.__qualname__:
            return TEXT_SOURCE_TABLE
        case NormalCellHeaderHandler.__qualname__:
            return TEXT_SOURCE_CELL  
        case _:
            return TEXT_SOURCE_UNKNOWN

# Two ways of making a smart tuple: NamedTuple is more recent in Python history
from collections import namedtuple
from typing import NamedTuple

class PointElementInfo (NamedTuple):
    # Class stores data about a point DGN element, such as a cell or text.
    # The text stores the cell name or, if a text element, the text string it contains.  
    elementId: np.int64
    levelId: np.int32
    text: str
    # Coordinates in world units e.g. metres, feet etc
    x: float
    y: float
    z: float
    source: str
    sourceId: np.int64 # The element ID of the container that holds the text.  Zero if a stand-alone text element
 
    def __str__(self):
        # Override the default __str__ function to print the data in a text instance.
        # i.e. Print in a useful format for users
        return f"'{self.text}', {str(self.elementId)}, {str(self.levelId)}, {str(self.x)}, {str(self.y)}, {str(self.z)}, '{self.source}', {str(self.sourceId)}"
    
    def __repr__(self):
        # Override the default __repr__ function to return a string representation of the
        # instance in a format that can be used to recreate the object.
        # i.e. Print in a format useful to developers
        return f"PointElementInfo({str(self.elementId)}, {str(self.levelId)}, '{self.text}', {str(self.x)}, {str(self.y)}, {str(self.z)}, '{self.source}', {str(self.sourceId)})"
       
    def __gt__(self, other):
        # Provide a comparison > operator so that lists of this class can be sorted.
        # I chose to sort by the text content, but you might want some other criterion
        if isinstance(other, PointElementInfo):
            # Compare two strings ignoring case using casefold()
            return self.text.casefold() > other.text.casefold()
        else:
            return NotImplemented
   
    def __eq__(self, other):
        # Provide an equality == operator.
        # I chose to compare the Element IDs, but you might want some other criterion
        if isinstance(other, PointElementInfo):
            return self.elementId == other.elementId
        else:
            return NotImplemented
        
    @classmethod
    def ColumnNames (cls)->list[str]:
        # Class method supplies a list of column names used, for example, in an Excel worksheet
       return ['Element ID', 'Level ID', 'Text', 'X', 'Y', 'Z', 'Source', 'Source ID']
       
if __name__ == "__main__":  # check if this script is being run directly (not imported as a module)
    vinfo = VersionInfo("PointElementInfo", 25, 2, 2, "Data extracted from a point element e.g. text, cell")
    MessageCenter.ShowDebugMessage(vinfo.brief, vinfo.verbose, False)
    # Test class using its __str__ and __repr__ magic methods
    pointInfo = PointElementInfo(1234, 5678, "word", 10.10, 11.11, 12.12, 'cell', 9876)
    print(pointInfo)
    print(repr(pointInfo))
    
    
 