# -*- coding: utf-8 -*-

'''
Created 21-Feb-2025
Utilities for Excel I/O.
Compose a fully-qualified file name for import or export.
Compose an Excel sheet name subject to Microsoft constraints.
'''
import os.path
from pathlib import Path
import pandas as pd 

from MSPyBentley import BeFileName
from MSPyMstnPlatform import MessageCenter, ISessionMgr
from la_solutions.version_info import VersionInfo
from la_solutions.configuration_vars import *
 
def CreateExcelSheetName (preamble, name)->str:
    '''
    Create an Excel sheet name from a preamble and a name, subject to Excel constraints.
    https://www.google.com/search?client=firefox-b-d&q=excel+valid+sheet+name
    '''
    MAX_EXCEL_SHEET_LABEL_LENGTH = 31
    labelName = f" '{name}' " # Trailing space after closing quote is intended
    if MAX_EXCEL_SHEET_LABEL_LENGTH < len(preamble) + len(name):
        #   Reduce preamble
        remainder = name[:MAX_EXCEL_SHEET_LABEL_LENGTH - len(name) - 2]
        return f"{preamble [:remainder]}{labelName}"
    else:
        return preamble + f"{labelName}"
   
class ExcelFile:
    '''
    Base class ExcelFile computes a fully-qualified file name from the DGN file name,
    and the passed file_path.
    You can replace components of the final path using the ReplaceXxx() methods.
    Named parts of file from MicroStationAPI BeFileName
    enum FileNameParts {Device=1, Directory=2, Basename=4, Extension=8, DevAndDir=(Device|Directory), NameAndExt=(Basename|Extension), All=(DevAndDir|NameAndExt) };
    '''
    def __init__(self, file_path, extension = "xlsx"):
        self.dgnFile = ISessionMgr.ActiveDgnFile  # Get the active DGN file from the MicroStation session manager
        self.modelName = ISessionMgr.ActiveDgnModelRef.GetDgnModel().GetModelName()
        # BeFileName inherits from WString, so expect conversions between Python str and Bentley WString
        self.beFileName = BeFileName()
        self.beFileName.BuildName ()
    @property
    def FolderExists(self)->bool:
        '''
        Test whether the path computed by this class really exists.
        If we're reading or writing a file, then the folder must exist.
        '''
        #return os.path.exists(os.path.abspath(self.filePath))
        
    @property
    def FileExists(self)->bool:
        '''
        Test whether the file computed by this class really exists.
        If we're reading a file, then the file must exist.
        '''
        return os.path.exists(self.FullPathName)
        
    @property
    def ModelName(self)->str:
        return self.modelName
         
    @property
    def FileNameWithoutExtension(self)->str:
        return self.baseName
 
    @property
    def Extension(self)->str:
        return str(self.beFileName.GetExtension())

    @property
    def FullPathName(self)->str:
        '''
        Returns the fully-qualified path to our file, including the file name.
        '''
        return self.beFileName.BuildName  

    @property
    def DirectoryName(self)->str:
        '''
        Returns the fully-qualified path to our file, excluding the file name.
        '''
        return str(GetDirectoryName( os.path.join(os.path.dirname(os.path.abspath(self.filePath)))))

    def ReplaceFilePath(self, newPath: str)->str:
        ''' Provides a new file path for self.filePath. '''
        self.filePath = newName
        return self.Path

    def ReplaceFileName(self, newName: str)->str:
        ''' Provides a new file name for self.filePath. '''
        self.baseName = newName
        return self.Path

    def ReplaceExtensionName(self, newExt: str)->str:
        ''' Provides a new file extension name for self.filePath. '''
        self.extension = newExt
        return self.Path

class ExcelOutputFile(ExcelFile):
    '''
    Class ExcelFile computes a file name from the DGN file name,
    folder indicated by configuration variable MS_REPORTS_OUT, and the Excel file extension
    '''
    def __init__(self):
        valid, path = GetReportFolder()
        super().__init__ (path)
        
    def WriteDataFrame (self, df: pd.DataFrame, sheetLabel: str)->bool:
        '''
        Wraps the curious logic required to write a DataFrame to an Excel workbook.
        '''
        if df.empty:
            msg = "DataFrame is empty: cannot create Excel file"
            MessageCenter.ShowErrorMessage(msg,msg, False)
            return False
        # Open the Excel file in write mode using pandas.
        # This fails if the file is already open in Excel.
        with pd.ExcelWriter(self.Path) as writer:
            # Write the DataFrame to the Excel file
            msg = f'Write to Excel sheet "{sheetLabel}"'
            MessageCenter.ShowDebugMessage(msg, msg, False)
            df.to_excel(writer, index = False, sheet_name = sheetLabel) # trailing space for Excel       
            # Get a reference to the workbook and worksheet
            workbook  = writer.book
            worksheet = writer.sheets[sheetLabel]
            worksheet.Visible = True
            # Save the workbook (and therefore the entire Excel file)
            workbook.save(self.Path)
            msg = f"Saved DataFrame to file '{self.BaseName}.{self.Extension}'"
            MessageCenter.ShowDebugMessage(msg, self.Path, False) 
            return True
        return False           

class ExcelInputFile(ExcelFile):
    '''
    Class ExcelFile computes a file name from the DGN file name,
    folder indicated by configuration variable MS_TABLE_INPUT, and the Excel file extension
    '''
    def __init__(self):
        valid, path = GetInputFolder()
        super().__init__ (path)
        self.df = None
    
    @property
    def DataFrame(self)->pd.DataFrame:
        '''
        On opening and reading the Excel worksheet, we assign the data to a pandas DataFrame.
        You can get that DataFrame using this property.
        '''
        return self.df
        
    def ReadExcelToDataFrame (self)->pd.DataFrame:
        '''
        On opening and reading the Excel worksheet, we assign the data to a pandas DataFrame.
        '''
        with pd.ExcelFile(self.Path) as reader:
            sheetNames = reader.sheet_names
            n = 0
            for sheetName in sheetNames:
                msg = f"Sheet[n] '{sheetName}'"
                n += 1
                MessageCenter.ShowDebugMessage(msg, msg, False)
            # Read the DataFrame from the Excel file
            # Next line works only if there is more than one worksheet in the Excel file
            #msg = f'Read from Excel sheet {sheetNames[1]}'
            #MessageCenter.ShowDebugMessage(msg, msg, False)        
            #frame = pd.read_excel(reader, sheetNames[1])
            msg = f'Read from Excel sheet {sheetNames[0]}'
            MessageCenter.ShowDebugMessage(msg, msg, False)        
            self.df = pd.read_excel(reader, sheetNames[0]) 
        return self.df       

    def DataFrameHasColumn (self, name: str)->bool:
        '''
        Verify that the DataFrame has named column.
        '''
        result = [col for col in self.df.columns if name in col]
        return 0 < len(result)
        
    def DataFrameHasXYColumns(self)->bool:
        '''
        Verify that the DataFrame has a X and Y columns, used
        to create a new DGN element.
        '''
        return self.DataFrameHasColumn("X") and self.DataFrameHasColumn("Y")
        
    def DataFrameHasElementId(self)->bool:
        '''
        Verify that the DataFrame has an Element ID column, used
        when replacing existing DGN text elements.
        '''
        return self.DataFrameHasColumn("Element ID")
        
class ExcelTableInputFile(ExcelFile):
    '''
    Class ExcelFile computes a file name from the DGN file name,
    folder indicated by configuration variable MS_TABLE_INPUT, and the Excel file extension
    '''
    def __init__(self):
        valid, path = GetTableInputFolder()
        super().__init__ (path)
        self.df = None

    @property
    def DataFrame(self)->pd.DataFrame:
        '''
        On opening and reading the Excel worksheet, we assign the data to a pandas DataFrame.
        You can get that DataFrame using this property.
        '''
        return self.df
        
    def ReadExcelToDataFrame (self)->pd.DataFrame:
        '''
        On opening and reading the Excel worksheet, we assign the data to a pandas DataFrame.
        '''
        with pd.ExcelFile(self.Path) as reader:
            sheetNames = reader.sheet_names
            n = 0
            for sheetName in sheetNames:
                msg = f"Sheet[n] '{sheetName}'"
                n += 1
                MessageCenter.ShowDebugMessage(msg, msg, False)
            # Read the DataFrame from the Excel file
            # Next line works only if there is more than one worksheet in the Excel file
            #msg = f'Read from Excel sheet {sheetNames[1]}'
            #MessageCenter.ShowDebugMessage(msg, msg, False)        
            #frame = pd.read_excel(reader, sheetNames[1])
            msg = f'Read from Excel sheet {sheetNames[0]}'
            MessageCenter.ShowDebugMessage(msg, msg, False)        
            self.df = pd.read_excel(reader, sheetNames[0]) 
        return self.df       
        
def Version():
    ''' Return the version of this app. '''
    return VersionInfo("Excel Utilities", 25, 2, 25, "Get file name for export or import")

if __name__ == "__main__":  # check if this script is being run directly (not imported as a module)
    vinfo = Version()
    MessageCenter.ShowInfoMessage(vinfo.brief, vinfo.verbose, False) 
    
   
    print ("Input Table file")
    f = ExcelTableInputFile()
    print (f.Path)
    msg = f"test path {f.FilePath}"
    MessageCenter.ShowDebugMessage(msg, msg, False) 
    if f.FileExists:
        msg = f"path {f.FilePath} exists"
        MessageCenter.ShowDebugMessage(msg, msg, False) 
    else:
        msg = f"path {f.FilePath} does not exist"
        MessageCenter.ShowErrorMessage(msg, msg, False) 
    msg = f"test file {f.Folder}"
    MessageCenter.ShowDebugMessage(msg, msg, False) 
    if f.FileExists:
        msg = f"folder {f.Folder} exists"
        MessageCenter.ShowDebugMessage(msg, msg, False) 
    else:
        msg = f"folder {f.Folder} does not exist"
        MessageCenter.ShowErrorMessage(msg, msg, False) 
    
    print ("Input file")
    f = ExcelTableInputFile()
    print (f.Path)    
    
    print ("Output file")
    f = ExcelOutputFile()
    print (f.Path)
    
    df = pd.DataFrame ({
        "calories": [420, 380, 390],
        "duration": [50, 40, 45]
            })
    print (df)
    excel = ExcelOutputFile()
    

        