# -*- coding: utf-8 -*-

'''
Created 11-Mar-2025
This is the implemention of the Pick Line class.
'''
from MSPyBentley import *
from MSPyBentleyGeom import DPoint3d, DPoint2d, ICurvePrimitive, CurveLocationDetail
from MSPyDgnPlatform import *
from MSPyMstnPlatform import *
from MSPyDgnView import *

from PyQt5.QtGui import QWindow

from FollowLine.follow_line_qt import Window

# Borrowed from dgn_elements
ELEM_HANDLER_GET_DESCR_MAX_LENGTH = 32      
def GetElementDescription   (element, 
                            string_length: int = ELEM_HANDLER_GET_DESCR_MAX_LENGTH, 
                            dgn_model: DgnModel = ISessionMgr.ActiveDgnModelRef.GetDgnModel())->str:
    '''
    Get the human-readable description of an element from an elementRef or ElementHandle.
    
    params:
    element [in] is an ElementHandle or ElementRef
    string_length [in] is the maximum length of the description string you want to accept; default to 32 
    dgn_model [in] is a DGN model that contains element; defaults to the active model
    '''
    description = WString()
    if isinstance (element, ElementHandle):
        handler = element.GetHandler()
        if handler is None:
            MessageCenter.ShowErrorMessage(f"GetElementDescription() ElementHandle is invalid", None, False)
        else:
            handler.GetDescription (element, description, string_length)
        
    elif isinstance (element, PersistentElementRef):
        eh = ElementHandle (element, dgn_model)
        handler = eh.GetHandler()       
        handler.GetDescription (eh, description, string_length)
        
    return str(description)

class cmdPickLineElement(DgnElementSetTool):
    '''
    Inherits from DgnElementSetTool.  Prompts user to pick a line element.
    'line element' means anything that converts to an open CurveVector.
    '''
    _ACCEPTABLE_TYPES = (ICurvePrimitive.eCURVE_PRIMITIVE_TYPE_Line,
                        ICurvePrimitive.eCURVE_PRIMITIVE_TYPE_LineString,
                        ICurvePrimitive.eCURVE_PRIMITIVE_TYPE_Arc,
                        ICurvePrimitive.eCURVE_PRIMITIVE_TYPE_CurveVector,
                        )

    def __init__(self, window, toolId):
        ''' Initialise this class and its base class DgnElementSetTool.  '''
        # Don't call super()
        DgnElementSetTool.__init__(self, toolId) # C++ base's __init__ must be called.
        self.m_isDynamics = False
        self.m_ev = None
        self.m_self = self # Keep self referenced
        self.m_window = window # The Qt window that started this command
        
    def _GetToolName(self, name):
        name = WString("Pick line")
        
    def _DoGroups(self):
        '''
        Pick single element, don't include graphic and named group members.
        '''
        return False 
        
    def _AllowSelection(self):
        ''' Selection set not allowed. '''
        return DgnElementSetTool.eUSES_SS_None 
        
    def _SetupForModify(self, ev, isDynamics):
        '''  '''
        self.m_isDynamics = isDynamics
        self.m_ev = ev
        
        return True
    
    def _is_acceptable_primitive_type(self, primitiveType, acceptable_types: list)->bool:
        '''
        Test this primitive type against a list of acceptable types.
        '''
        return primitiveType in acceptable_types
            
    def _OnElementModify(self, eeh)->int:
        '''
        Since we've already validated the element in OnPostLocate and don't support groups/selection sets 
        we don't have to check it again here.
        '''
        curve = ICurvePathQuery.ElementToCurveVector(eeh)
        primitive = curve[0]
        location = CurveLocationDetail()

        locatePoint = DPoint3d()
        self._GetAnchorPoint(locatePoint)

        ''' Get curve location detail from accept point. '''
        if not curve.ClosestPointBounded(locatePoint, location):
            return BentleyStatus.eERROR
       
        primitiveType = primitive.GetCurvePrimitiveType()
        #print (f"_OnElementModify primitiveType '{primitiveType}'")
        if self._is_acceptable_primitive_type(primitiveType, self._ACCEPTABLE_TYPES):
            self.m_window.set_picked_element_id(eeh.GetElementId())
            #label: QLabel = self.m_window.find_lbl_element_id()
            #label.setText(str(eeh.GetElementId()))
            print(f"cmdPickLine picked {GetElementDescription(eeh)} ID {eeh.GetElementId()}")
            
            return BentleyStatus.eSUCCESS
        # Return ERROR to signify no change.    
        return BentleyStatus.eERROR
        
    def _OnPostLocate(self, path, cantAcceptReason)->bool:
        '''
        Called during a locate to allow tool to filter which elements can be selected.  
        '''
        #msg = "_OnPostLocate"
        #MessageCenter.ShowDebugMessage(msg, msg, False) 
       
        if not DgnElementSetTool._OnPostLocate(self, path, cantAcceptReason):
            #msg = "DgnElementSetTool._OnPostLocate (super) returned false"
            #MessageCenter.ShowErrorMessage(msg, msg, False) 
            return False

        eh = ElementHandle(path.GetHeadElem(), path.GetRoot())
        curve = ICurvePathQuery.ElementToCurveVector(eh)

        # Accept elements that are open paths with one or more linear segments (e.g. line or linestring).
        if curve == None or (not curve.IsOpenPath()):
            # This produces a  lot of messages as you move the cursor
            cantAcceptReason = "Not an open element"
            MessageCenter.ShowErrorMessage(cantAcceptReason, cantAcceptReason, False) 
            return False

        primitiveType = curve.HasSingleCurvePrimitive()
        #print (f"_OnPostLocate: primitiveType '{primitiveType}'")
        if self._is_acceptable_primitive_type(primitiveType, self._ACCEPTABLE_TYPES):
            #print(f"_OnPostLocate accepted primitive type {primitiveType}")                                 
            return True
       
        cantAcceptReason = f"primitiveType={primitiveType} Not an open element"
        msg = cantAcceptReason
        MessageCenter.ShowErrorMessage(msg, msg, False) 
        return False
     
    def _SetupAndPromptForNextAction(self):
        '''
        An opportunity to display a message to your user
        '''
        msg = 'Pick Line or Arc'
        NotificationManager.OutputPrompt(msg)
        
    def _OnRestartTool(self):
        '''    
        Install a new instance of the tool. Will be called in response to external events
        such as undo or by the base class from _OnReinitialize when the tool needs to be
        reset to it's initial state.
        '''
        InstallNewInstance(self.m_window, 0)
    
def InstallNewInstance(window = None, toolId = 0):
    '''
    Method to create and install a new instance of the tool. If InstallTool returns ERROR,
    the new tool instance will be freed/invalid. 
    '''
    print("cmd_pick_line.InstallNewInstance()")
    tool = cmdPickLineElement(window, toolId)
    tool.InstallTool()

if __name__ == "__main__":  # check if this script is being run directly (not imported as a module)
    msg = "Implementation of state machine for Pick Line"
    MessageCenter.ShowDebugMessage(msg, msg, False)

    InstallNewInstance()
