# -*- coding: utf-8 -*-

'''
Created 26-Jan-2025
Updated 10-May-2025
Utility functions that operate on DGN objects.
'''

from MSPyBentley import *
from MSPyBentleyGeom import *
from MSPyDgnPlatform import *
from MSPyMstnPlatform import *

# See: https://bentleysystems.service-now.com/community?id=community_question&sys_id=3950e8461b6612147171ff7e034bcb58
# Used with ChildElemIter
EXPOSE_CHILDREN_COUNT = ExposeChildrenReason(100)
EXPOSE_CHILDREN_QUERY = ExposeChildrenReason(200)
EXPOSE_CHILDREN_EDIT  = ExposeChildrenReason(300)

ELEM_HANDLER_GET_DESCR_MAX_LENGTH = 32      
def GetElementDescription   (element, 
                            string_length: int = ELEM_HANDLER_GET_DESCR_MAX_LENGTH, 
                            dgn_model: DgnModel = ISessionMgr.ActiveDgnModelRef.GetDgnModel())->str:
    '''
    Get the human-readable description of an element from an elementRef or ElementHandle.
    
    params:
    element [in] is an ElementHandle or ElementRef
    string_length [in] is the maximum length of the description string you want to accept; default to 32 
    dgn_model [in] is a DGN model that contains element; defaults to the active model
    '''
    description = WString()
    if isinstance (element, ElementHandle):
        handler = element.GetHandler()
        if handler is None:
            MessageCenter.ShowErrorMessage(f"GetElementDescription() ElementHandle is invalid", None, False)
        else:
            handler.GetDescription (element, description, string_length)
        
    elif isinstance (element, PersistentElementRef):
        eh = ElementHandle (element, dgn_model)
        handler = eh.GetHandler()       
        handler.GetDescription (eh, description, string_length)
        
    return str(description)

def GetElementById(element_id: int, dgn_model: DgnModel = ISessionMgr.ActiveDgnModelRef.GetDgnModel())->[bool, ElementHandle]:
    '''
    Get an element by its ID in the supplied DGN model.
    Returns: tuple (boolean: valid, Element Handle)
    
    params:
    element_id [in] the ID of the element to seek
    dgn_model [in] the DGN model that contains the element; default to the active model.
    '''
    eh = ElementHandle(element_id, dgn_model)  
    valid = eh.IsValid()
    if not valid:
        msg = f"Element ID {element_id} is invalid"
        MessageCenter.ShowErrorMessage(msg, msg, False)
    
    return (valid, eh)

def GetUorsPerMeter (dgn_modelRef: DgnModelRef = ISessionMgr.ActiveDgnModelRef)->float:
    '''Get the ratio of units-of-resolution per meter for specified model.
    If you don't specify a model then the active model is assumed.'''
    dgn_model = dgn_modelRef.GetDgnModel()
    modelInfo = dgn_model.GetModelInfo()
    # Convert MicroStation units-of-resolution to meters
    uors: float = modelInfo.GetUorPerMeter()
    return uors

def GetUorsPerMaster (dgn_modelRef: DgnModelRef = ISessionMgr.ActiveDgnModelRef)->float:
    '''Get the ratio of units-of-resolution per master unit for specified model.
    If you don't specify a model then the active model is assumed.'''
    # Convert MicroStation units-of-resolution to master units
    uors: float = ModelRef.GetUorPerMaster(dgn_modelRef)
    return uors

def VerifyElementType (eh: ElementHandle, handlers: tuple)->bool:
    ''' Check that an element's class, represented by its element handler, 
    matches at least one type in the handlers tuple '''
    handler = eh.GetHandler()    
    return isinstance(handler, handlers)

def GetScanRange (eh: ElementHandle)->ScanRange:
    ''' Get an element's range as a ScanRange. A ScanRange struct has Int64 members...
    Int64  xlowlim       
    Int64  ylowlim       
    Int64  zlowlim       
    Int64  xhighlim       
    Int64  yhighlim       
    Int64  zhighlim 
    
    If in doubt, check IsGraphical before calling this. '''
    return eh.GetElement().hdr.dhdr.range
 
def is_element_in_range(eh: ElementHandle, scan_range: ScanRange)->bool:
    ''' Test if an element lies within a specified ScanRange. '''
    element_range = eh.GetElement().hdr.dhdr.range
    if element_range.xlowlim < scan_range.xlowlim: return False
    if element_range.ylowlim < scan_range.ylowlim: return False
    if element_range.zlowlim < scan_range.zlowlim: return False
    if element_range.xhighlim > scan_range.xhighlim: return False
    if element_range.yhighlim > scan_range.yhighlim: return False
    if element_range.zhighlim > scan_range.zhighlim: return False
    return True
    
def IsInvisible (eh: ElementHandle)->bool:
    ''' Determine whether a DGN element is visible. 
    
    Wraps the DGN C-style element structure. 
    
    If in doubt, check IsGraphical before calling this.  '''
    return eh.GetElement().hdr.dhdr.props.b.invisible
    
def IsVisible (eh: ElementHandle)->bool:
    ''' Determine whether a DGN element is visible. 
    
    Wraps the DGN C-style element structure. '''
    return not IsInvisible(eh)
    
def IsGraphical (eh: ElementHandle)->bool:
    ''' Determine whether a DGN element is graphical. 
    
    Wraps the DGN C-style element structure. '''
    return eh.GetElement().ehdr.isGraphics
    
def IsDeleted (eh: ElementHandle)->bool:
    ''' Determine whether a DGN element is deleted. '''
    return eh.GetElement().ehdr.deleted
    
def GetElementClass (eh: ElementHandle)->DgnElementClass:
    ''' Return the DGN class (i.e. primary, construction etc) of a DGN element. 
    
    Wraps the DGN C-style element structure. 
    
    If in doubt, check IsGraphical before calling this. '''
    return eh.GetElement().hdr.dhdr.props.b.elementClass
    
def IsPrimaryClass (eh: ElementHandle)->bool: 
    ''' Determine whether a DGN element is primary DGN class. '''
    return GetElementClass(eh) == DgnElementClass.ePrimary

def IsConstructionClass (eh: ElementHandle)->bool:  
    ''' Determine whether a DGN element is primary DGN class. '''
    return GetElementClass(eh) == DgnElementClass.eConstruction    
    
def IsVisibleGraphicalElement (eh: ElementHandle)->bool:
    ''' Determine that this is a visible graphical DGN element. '''
    return IsGraphical(eh) and IsVisible(eh)
    
def GetElementLevelId (eh: ElementHandle)->int:
    ''' Get the level ID of a graphic element. 
    
    Wraps the DGN C-style element structure. '''
    if IsGraphical(eh):
        return eh.GetElement ().msElement.ehdr.level
    #else
    return 0
 
def IsClosedElement(eh : ElementHandle)->bool:
    ''' Determine whether this is a visible closed element. '''
    if (eh.IsValid() and IsVisibleGraphicalElement(eh)): 
        path : CurveVector = ICurvePathQuery.ElementToCurveVector(eh)
        if path is None:
            return False
        return path.IsClosedPath()
    return False
    
def HasItem(eh : ElementHandle, libName: str, itemName: str)->bool:
    ''' Determine whether an element has a named Item in a named Item Type Library. '''
    if (eh.IsValid()):
        host: CustomItemHost = CustomItemHost(eh)
        if host.GetCustomItem(libName, itemName):
            return True
    
    return False    