import sys
import PyQt5
from PyQt5 import QtCore
from PyQt5.QtCore import QTimer, Qt, QObject, pyqtSlot
from PyQt5.QtGui import QFont, QWindow, QValidator, QDoubleValidator, QIntValidator
from PyQt5.QtWidgets import (
    QApplication,
    QComboBox, 
    QGridLayout,
    QGroupBox,
    QHBoxLayout,
    QLabel,
    QLineEdit,
    QMainWindow,
    QPushButton,
    QStackedLayout,
    QVBoxLayout,
    QWidget,
)

# Two imports for testing command callback suggestion
import time
import win32gui

from MSPyBentley import *
from MSPyBentleyGeom import *
from MSPyDgnPlatform import *
from MSPyDgnView import *
from MSPyMstnPlatform import *

from la_solutions.version_info import VersionInfo
#from la_solutions.qt_window_base_class import WindowBase
import la_solutions.cell_library as cell_utilities

import FollowLine
from FollowLine.command_requests import CommandRequests
from FollowLine.import_invoke_function import import_function, invoke_function

#
# Place cells along line UI for example Python app.
#
# This demonstrates a PyQt5 UX window containing a small number of user-interface widgets.
# The window uses a base class that contains a timer whose purpose is to trigger a windows message pump.
# Registering a timer with a 0 start interval will cause the timer method to execute every time the Qt Windows event loop finishes. 
# Calling PyCadInputQueue.PythonMainLoop from the timer will allow MicroStation's input loop to execute allowing use of the 
# MicroStation UX whilst the Qt window is displayed.
#
# Unfortunately there is a conflict between commands placed in MicroStation's input queue and Windows messaging.
# Consequently, it's impossible (2024) to have both MicroStation commands and a Python UI in the same application.
#
# In this example we choose to provide a Qt UI but no MicroStation commands.

#class Window(WindowBase):
class Window(QMainWindow):
    ''' Our window inherits from base class WindowBase, which in turn inherits from QMainWindow. '''
    def __init__(self, vinfo: VersionInfo):
        super().__init__()
        self._storedWinId = self.winId()
        self._loop = QtCore.QEventLoop()
        self._versionInfo = vinfo
        self._picked_element_id = 0
        self._BUTTON_TOP = 10
        self._BUTTON_LEFT = 10
        self._BUTTON_WIDTH  = 180
        self._BUTTON_HEIGHT = 100
        self._CBO_CELL_NAME = "cboCellName"
        # self._CMD_xxx is the name of a function to be invoked when a Qt button or menu is activated
        self._CMD_PICK_LINE = "InstallNewInstance"
        # self._MODULE_xxx is the name of the Python module that contains the command function 
        self._MODULE_PICK_LINE = "FollowLine.cmd_pick_line"
        self._cmd_pick_line_requested = False
        self._CMD_EXECUTE = "execute"
        self._MODULE_EXECUTE = "FollowLine.cmd_execute"
        self._cmd_execute_requested = False
        self._LBL_ELEMENT_ID = 'lblElementId'
        self._LBL_POINT_COUNT = 'lblPointCount'
        self._TXT_INTERVAL = 'txtInterval'
        self._TXT_SCALE = 'txtScale'
        self.initUI()
        # Give the window a unique name for easier identification if needed for cleanup.
        self.setObjectName("dlgPlaceCellsAlongLine")
        # A list of command requests for this AddIn.  A request is typically a function call, rather than a MicroStation key-in command
        requests = {(self._MODULE_PICK_LINE, self._CMD_PICK_LINE): False, (self._MODULE_EXECUTE, self._CMD_EXECUTE): False}
        self._requests = CommandRequests(requests)
        print(self._requests)

    @pyqtSlot()
    def queue_action_pick_line(self):
        ''' Start the cmd_pick_line state engine. '''
        msg = "queue_action_pick_line"
        MessageCenter.ShowDebugMessage(msg, msg, False) 
        print(msg)
        #print(cmdPickLineElement.InstallNewInstance)
        # Uncomment the following statement to crash MicroStation
        #cmdPickLineElement.InstallNewInstance()
        #self._pending_tool_call = cmdPickLineElement.InstallNewInstance
        #self._requests._clear_requests()
        #self._requests[self._CMD_PICK_LINE] = True
        self._cmd_pick_line_requested = True
     
    @property
    def picked_element_id(self)->int:
        return self._picked_element_id
        
    @property
    def cell_name(self)->str:
        cbo: QComboBox = self.find_cbo_cell_name()
        name = cbo.currentText()
        return name

    @property
    def interval(self)->int:
        txt: QLineEdit = self.find_txt_interval()
        content = txt.text()
        return int(content)
        
    @pyqtSlot()
    def queue_action_execute(self):
        ''' Send a key-in string to MicroStation. '''
        msg = "queue_action_execute"
        MessageCenter.ShowDebugMessage(msg, msg, False) 
        #self._requests._clear_requests()
        #self._requests[self._CMD_EXECUTE] = True
        self._cmd_execute_requested = True
       
    def makeLabel(self)->QLabel:
        ''' Create a label widget. '''
        label = QLabel(self)
        #   Labels can understand HTML markup
        label.setText(f"<h3>{self._versionInfo.verbose}</h3>")
        return label

    def makePointCountLabel(self)->QLabel:
        ''' Create a label widget. '''
        label = QLabel(self)
        #   Labels can understand HTML markup
        label.setText("Points Created: ")
        label.setObjectName(self._LBL_POINT_COUNT)
        return label

    def makePickButton(self)->QPushButton:
        ''' Create a push button to start the Pick command. '''
        btn = QPushButton("Pick Line")
        btn.setFixedWidth(self._BUTTON_WIDTH)
        btn.clicked.connect(self.queue_action_pick_line)
        # Not allowed
        # btn.clicked(PyCadInputQueue.SendKeyin(KeyIn.PICK_LINE))
        return btn
    
    def makeExecuteButton(self)->QPushButton:
        ''' Create a push button to start the Pick command. '''
        btn = QPushButton("Execute")
        #btn.setGeometry(self._BUTTON_LEFT, self._BUTTON_TOP, self._BUTTON_WIDTH, self._BUTTON_HEIGHT)
        #btn.resize(self._BUTTON_WIDTH, self._BUTTON_HEIGHT)
        btn.setFixedWidth(self._BUTTON_WIDTH)
        btn.clicked.connect(self.queue_action_execute)
        # Not allowed
        # btn.clicked(PyCadInputQueue.SendKeyin("KeyIn.EXECUTE"))
        return btn
        
    def makeLaSolutionsLabel (self)->QLabel:
        label = QLabel(self, alignment=Qt.AlignmentFlag.AlignBottom)
        label.setText("<h3>LA Solutions</h3>")
        # Display the label using a style-sheet to obtain CSS font attributes
        label.setStyleSheet('color: teal; font-family: Times New Roman; font-size: 16pt')
        return label

    def makeVersionLabel (self, vinfo: VersionInfo)->QLabel:           
        label = QLabel(self, alignment=Qt.AlignmentFlag.AlignRight | Qt.AlignmentFlag.AlignBottom)
        label.setText(vinfo.brief)
        return label      
    
    def makeScaleLabel (self)->QLabel:           
        label = QLabel(self, alignment=Qt.AlignmentFlag.AlignRight | Qt.AlignmentFlag.AlignBottom)
        label.setText("Scale:")
        return label      
    
    def makeScaleText(self)->QLineEdit:
        txt =  QLineEdit(self)
        txt.setFixedWidth(self._BUTTON_WIDTH)
        # Check that user enters a valid number for the cell scale.
        _MIN_VAL = .1
        _MAX_VAL = 100
        _PRECISION = 2
        _INITIAL_SCALE = 0.1
        txt.setPlaceholderText("Scale must be a positive number .1..100")
        txt.setValidator(QDoubleValidator(_MIN_VAL, _MAX_VAL, _PRECISION, txt))
        txt.setObjectName(self._TXT_SCALE)
        txt.setText(str(_INITIAL_SCALE))
        return txt
    
    def get_scale (self)->float:
        ''' Return the numeric scale from the _TXT_SCALE text field. '''
        txt = self.find_named_widget(self._TXT_SCALE)
        scale = float(txt.text())
        return scale
    
    def makeIntervalLabel (self, text)->QLabel:           
        label = QLabel(self, alignment=Qt.AlignmentFlag.AlignRight | Qt.AlignmentFlag.AlignBottom)
        label.setText(text)
        return label      
    
    def makeIntervalText(self)->QLineEdit:
        txt =  QLineEdit(self)
        txt.setFixedWidth(self._BUTTON_WIDTH)
        # Check that user enters a valid number for the interval between cells.
        # In this example, the interval must lie in the range 1..100
        _MIN_VAL = 1
        _MAX_VAL = 100
        _INITIAL_INTERVAL = 2
        txt.setPlaceholderText("Interval must be a positive number 1..100")
        txt.setValidator(QIntValidator(_MIN_VAL, _MAX_VAL, txt))
        txt.setObjectName(self._TXT_INTERVAL)
        txt.setText(str(_INITIAL_INTERVAL))
        return txt
        
    def makePickedIdLabel (self, text)->QLabel:           
        label = QLabel(self, alignment = Qt.AlignmentFlag.AlignRight | Qt.AlignmentFlag.AlignBottom)
        label.setObjectName(self._LBL_ELEMENT_ID)
        label.setText(text)
        return label      

    def find_named_widget(self, name)->QWidget: 
        '''
        Search children of this QWindow for the named QWidget.
        '''
        widget: QWidget = self.findChild(QWidget, name)
        if widget == None:
            print(f"Widget '{name}' not found")
        return widget
    
    def find_lbl_point_count(self)->QLabel:
        '''
        Return the named QLabel.
        '''
        return self.find_named_widget(self._LBL_POINT_COUNT)

    def set_lbl_point_count(self, n: int):
        lbl = self.find_lbl_point_count()
        lbl.setText(f"Point Count: {n}")
        
    def find_lbl_element_id(self)->QLabel:
        '''
        Return the named QLabel.
        '''
        return self.find_named_widget(self._LBL_ELEMENT_ID)
    
    def find_cbo_cell_name(self)->QComboBox:
        '''
        Return the named QComboBox.
        '''
        return self.find_named_widget(self._CBO_CELL_NAME)
    
    def find_txt_interval(self)->QLineEdit:
        '''
        Return the named QLineEdit.
        '''
        return self.find_named_widget(self._TXT_INTERVAL)
    
    def set_picked_element_id(self, id: int):
        '''
        Lets an external module set our stored Element ID.
        On successful receipt of the value, update the label Element ID.
        '''
        if id > 0:
            self._picked_element_id  = id
            lbl = self.find_lbl_element_id()
            lbl.setText(str(self._picked_element_id))
        else:
            msg = "Picked element ID must be a positive integer"
            print(msg)
            
    def makeCellNameLabel (self, text)->QLabel:           
        label = QLabel(self, alignment = Qt.AlignmentFlag.AlignRight | Qt.AlignmentFlag.AlignBottom)
        label.setText(text)
        return label      
    
    def makeCellNameText(self)->QLineEdit:
        txt =  QLineEdit(self)
        txt.setFixedWidth(self._BUTTON_WIDTH)
        return txt
    
    def makeCellNameCombo(self)->QComboBox:
        cbo = QComboBox(self)
        (valid, dgnFile) = cell_utilities.get_active_cell_library()
        if valid:
            cbo.addItems(cell_utilities.enumerate_cell_library(dgnFile))
        else:
            cbo.addItem('No active cell library')
        cbo.setObjectName(self._CBO_CELL_NAME)
        return cbo
    
    def _set_requested_flags(self):
        ''' This rather clunky logic is used to avoid problems with the QWidget actions interfering with MicroStation Python function calls.'''
        if self._cmd_pick_line_requested:
            #print("_cmd_pick_line_requested")
            self._requests.set_requested(self._CMD_PICK_LINE)
            self._cmd_pick_line_requested = False
        if self._cmd_execute_requested:
            #print("_cmd_execute_requested")
            self._requests.set_requested(self._CMD_EXECUTE)
            self._cmd_execute_requested = False
        
    def ms_mainLoop(self):
        """
        Custom main loop that pumps both Qt and MicroStation events,
        and executes any deferred tool-start calls in a safe context.
        This loop is the key to letting the UI and tool coexist.
        """
        #while self.isVisible():
        while win32gui.IsWindow(self._storedWinId):
            # Standard pump for a hybrid Qt/MicroStation application
            #QApplication.instance().processEvents()
            self._loop.processEvents()
            PyCadInputQueue.PythonMainLoop()
            #print(f"requests: {self._requests}")
            self._set_requested_flags()
                
            # Check if a tool start was requested by a QWidget action
            ((mod, cmd), flag) = self._requests.have_request()
            if flag:
                # If yes, run the stored function (e.g., InstallNewInstance)
                print(f"requested '{mod}:{cmd}'")
                invoke_function(mod, cmd, self)
                # Clear the flag so it only runs once per click
                self._requests.clear_requests()
                
            time.sleep(0.001)

    def initUI(self):
        ''' Initialise the PyQt user interface. '''
        self.setWindowTitle(self._versionInfo.brief)
        self.setGeometry(100, 200, 720, 320)
        outer_layout = QGridLayout()
        inner_layout = QGridLayout()
        # Row 0
        outer_layout.addWidget(self.makeLabel(), 0, 0)
        inner_layout.addWidget(self.makePickButton(), 0, 1)
        inner_layout.addWidget(self.makeExecuteButton(), 0, 3)
        # Row 1
        inner_layout.addWidget(self.makePickedIdLabel("Element ID"), 1, 1)
        # Row 2
        inner_layout.addWidget(self.makeIntervalLabel("Interval:"), 2, 0)
        inner_layout.addWidget(self.makeIntervalText(), 2, 1)
        inner_layout.addWidget(self.makeCellNameLabel("Cell Name:"), 2, 2)
        #inner_layout.addWidget(self.makeCellNameText(), 2, 3)
        inner_layout.addWidget(self.makeCellNameCombo(), 2, 3)
        # Row 3
        inner_layout.addWidget(self.makePointCountLabel(), 3, 1)
        inner_layout.addWidget(self.makeScaleLabel(), 3, 2)
        inner_layout.addWidget(self.makeScaleText(), 3, 3)        
        
        outer_layout.addLayout(inner_layout, 1, 0)
        outer_layout.addWidget(self.makeLaSolutionsLabel(), 2, 0)
        outer_layout.addWidget(self.makeVersionLabel(self._versionInfo), 2, 1)

        widget = QWidget()
        widget.setLayout(outer_layout)
        self.setCentralWidget(widget)
                    
    @classmethod
    def Run(cls, vinfo: VersionInfo):
        ''' Start PyQt and the main processing loop. 
        
        Defined as a class method.  Call using Window.Run() from __main__. '''
        app = QApplication([])
        w = Window(vinfo)
        w.show()
        w.ms_mainLoop()

if __name__ == "__main__":  # check if this script is being run directly (not imported as a module)
    Run()
