# -*- coding: utf-8 -*-

'''
Created 21-Jul-2025
This is the implemention of the Execute class for place cells along line.
Given an element ID, it extracts the geometry of that feature and then places the
named cell at stated intervals along that linear feature.
'''
from MSPyBentley import *
from MSPyBentleyGeom import DRay3d, DVec3d, DPoint3d, DPoint2d, ICurvePrimitive, CurveLocationDetail, CurveLocationDetailArray, DoubleArray, RotMatrix
from MSPyDgnPlatform import *
from MSPyMstnPlatform import *
from MSPyDgnView import *

from PyQt5.QtGui import QWindow
from PyQt5.QtWidgets import (
    QApplication,
    QComboBox, 
    QLabel,
    QLineEdit,
    QMainWindow,
    QPushButton,
    QWidget,
)

import math

import la_solutions.cell_library as cell_utilities

class cmdExecute():
    '''
    Given the ID of a linear element, place a cell at regular intervals along that line.
    '''
    def __init__(self, window: QWindow):
        ''' Initialise this class.  '''
        self.picked_element_id = window.picked_element_id
        self.interval = window.interval
        self.cell_name = window.cell_name
        self.window = window
        window.set_lbl_point_count(0)
        self.scale = window.get_scale()
        print(f"cmdExecute Line ID = {self.picked_element_id} interval={self.interval} scale={self.scale} cell '{self.cell_name}'")

    def is_valid_element(self):
        ''' Check that the element ID is valid.
        You may want to go further and verify that the ID is that of a linear element. '''
        return self.picked_element_id > 0
    
    def is_valid_cell_name(self):
        ''' Check that the supplied cell name is valid and available in a cell library. '''
        SEARCH_ALL = 1  # 0=no, 1=yes w/o messages, 2=yes w/ messages
        retVal, library = Cell.FindCell(None, self.cell_name, SEARCH_ALL)
        return BentleyStatus.eSUCCESS == retVal
 
    def is_valid_interval(self):
        ''' Check that the supplied interval is valid.  It must be a positive number. '''
        return self.interval > 0.0
        
    def requirements_met(self)->bool:
        ''' Check that user has supplied the necessary parameters to place cells along a line.  '''
        if  (self.is_valid_element()
            and
            self.is_valid_interval()
            and
            self.is_valid_cell_name()):           
            return True
        return False
        
def execute(window: QWindow):
    ''' for testing '''
    executor = cmdExecute(window)
    if executor.requirements_met():
        print ("Proceed with cell placement")
        cell_count = place_cells_along_line (executor.picked_element_id, executor.cell_name, None, executor.interval, executor.scale)
        window.set_lbl_point_count (cell_count)
        PyCommandState.StartDefaultCommand()
        print(f"Placed cells at {cell_count} points")
    else:
        msg = "Requirements not met for place cell along element"
        print(msg)

def get_tangent_at_point(detail: CurveLocationDetail)->RotMatrix:
    ''' Compute the tangent at a point on a line and return its orientation as a rotation matrix. '''
    vr: ValidatedDRay3d = detail.PointAndUnitTangent()
    ray: DRay3d = vr.Value()
    direction: DVec3d = ray.direction
    x_axis = DVec3d.From (1.0, 0.0, 0.0)
    angle = direction.AngleToXY(x_axis)
    _Z_AXIS = 2
    rotation = RotMatrix.FromAxisAndRotationAngle(_Z_AXIS, angle) 
    # Invert the rotation matrix so the cell is counter-rotated to match the line tangent
    rotation.Invert()
    return rotation   

# Generated mostly by MicroStation Python Assistant

def place_cells_along_line(line_element_id, cell_name, cell_library_path, interval, scale)->int:
    """
    Places cells at regular intervals along a line element in the active model.

    :param line_element_id: The element ID of the line along which to place cells.
    :param cell_name: The name of the cell to place.
    :param cell_library_path: The full path to the cell library (.cel) file.
    :param interval: The interval (in master units) at which to place cells.
    """
    ACTIVEMODEL = ISessionMgr.ActiveDgnModelRef
    dgnModel = ACTIVEMODEL.GetDgnModel()
    modelInfo = dgnModel.GetModelInfo()
    __, uorPerMast = modelInfo.StorageUnit.ConvertDistanceFrom(modelInfo.UorPerStorage, modelInfo.MasterUnit)

    # Get the line element as an EditElementHandle
    line_eeh = EditElementHandle(line_element_id, dgnModel)
    if not line_eeh.IsValid():
        print("Invalid line element ID.")
        return

    # Convert the element to a curve vector
    curve_vector = ICurvePathQuery.ElementToCurveVector(line_eeh)
    if curve_vector is None:
        print("Failed to convert element to curve vector.")
        return

    # Attach the cell library if required, otherwise use active library
    #cell_library = BeFileName(cell_library_path)
    #Cell.AttachLibrary(fileName=BeFileName(), inputName=cell_library, defaultDir=os.path.dirname(cell_library_path), fromKeyin=0)

    # Find the cell in the library
    _SEARCH_ALL = 1  # 0=no, 1=yes w/o messages, 2=yes w/ messages
    retVal, library = Cell.FindCell(None, cell_name, _SEARCH_ALL)
    if BentleyStatus.eSUCCESS != retVal:
        print(f"Error finding cell '{cell_name}' in library")
        return
    
    # Calculate points along the curve at the specified interval
    total_length = curve_vector.Length()
    interval_uor = interval * uorPerMast
    num_points = int(total_length // interval_uor) + 1

    print(f"total_length={total_length} N points={num_points}")
    
    distances = DoubleArray()
    for i in range(num_points):
        distances.append(i * interval_uor)

    points = CurveLocationDetailArray()
    curve_vector.AddSpacedPoints(distances, points)

    # Place the cell at each point
    for point in points:
        origin = point.point
        scale3d = DPoint3d(scale, scale, scale)
        tangent = get_tangent_at_point(point)
        try:
            Cell.PlaceCell(
                origin=origin,
                scale=scale3d,
                trueScale=True,
                rotMatrix=tangent,
                attributes=0,
                ggroup=0,
                relativeMode=False,
                baseLevel=0,
                sharedFlag=0,
                cellName=cell_name,
                library=library
            )
        except Exception as e:
            print(f"Error placing cell: {e}")
    return num_points
    
if __name__ == "__main__":  # check if this script is being run directly (not imported as a module)
    msg = "Implementation of place cell along line"
    MessageCenter.ShowDebugMessage(msg, msg, False)

    cmdExecute(123, 3.3, 'a_cell')

