# -*- coding: utf-8 -*-

'''
Created 11-Feb-2025
Utility functions that format floating-point numbers.
'''

from MSPyBentley import *
from MSPyBentleyGeom import *
from MSPyDgnPlatform import *
from MSPyMstnPlatform import *

def GetUorsPerMaster (dgnModel: DgnModel = ISessionMgr.ActiveDgnModelRef.GetDgnModel())->float:
    '''Get the ratio of units-of-resolution per master for specified model.
    Convert MicroStation units-of-resolution to master units
    '''
    uors: float = dgnModel.GetUorPerMaster()
    return uors

def GetUorsPerMeter (dgnModel: DgnModel = ISessionMgr.ActiveDgnModelRef.GetDgnModel())->float:
    '''
    Get the ratio of units-of-resolution per meter for specified model.
    If you don't specify a model then the active model is assumed.
    '''
    modelInfo = dgnModel.GetModelInfo()
    # Convert MicroStation units-of-resolution to meters
    uors: float = modelInfo.GetUorPerMeter()
    return uors

def DPoint2dToString (point: DPoint3d, dgnModel: DgnModel = ISessionMgr.ActiveDgnModelRef.GetDgnModel(), precision: PrecisionFormat = PrecisionFormat.eDecimal3Places)->str:       
    ''' Format a DPoint3d in string format.  Scales from Units of Resolution (UORs) to display units.
    Decimal precision may be specified (e.g. 1.23, 1.234, 1.2345 etc).
    '''
    formatter: PointFormatter = PointFormatter.Create(dgnModel, False)
    ds: DistanceFormatter = formatter.GetDistanceFormatter()
    ds.SetPrecision(precision)
    formatter.SetDistanceFormatter(ds)
    return f"XY={formatter.ToString(point)}"
    
def DistanceToString (distance: float, 
                        dgnModel: DgnModel = ISessionMgr.ActiveDgnModelRef.GetDgnModel(),
                            precision: PrecisionFormat = PrecisionFormat.eDecimal3Places)->str:       
    ''' Format a distance in string format.  Scales from Units of Resolution (UORs) to display units.
    Decimal precision may be specified (e.g. 1.23, 1.234, 1.2345 etc).
    '''
    ds: DistanceFormatter = DistanceFormatter.Create(dgnModel, False)
    ds.SetPrecision(precision)
    return f"XY={ds.ToString(distance)}"

def PrintDPointToMessageBox (comment: str, point: DPoint3d, dgnModel: DgnModel = ISessionMgr.ActiveDgnModelRef.GetDgnModel()):
    ''' Format a DPoint and forward to MicroStation's Message Centre '''
    msg = f"{comment} {DPoint2dToString(point, dgnModel, PrecisionFormat.eDecimal3Places)}"
    MessageCenter.ShowDebugMessage(msg, msg, False)
    
def PrintDistanceToMessageBox (comment: str, distance: float, dgnModel: DgnModel = ISessionMgr.ActiveDgnModelRef.GetDgnModel()):
    ''' Format a distance and forward to MicroStation's Message Centre '''
    msg = f"{comment} {DistanceToString(distance, dgnModel, PrecisionFormat.eDecimal3Places)}"
    MessageCenter.ShowDebugMessage(msg, msg, False)
    

