# -*- coding: utf-8 -*-

'''
Created 04-Mar-2025
See VersionInfo in __main__ for current update
'''
import os
from pathlib import Path

from MSPyBentley import *
from MSPyBentleyGeom import *
from MSPyDgnPlatform import *
from MSPyMstnPlatform import *

from la_solutions.configuration_vars import GetReportFolder
from la_solutions.version_info import VersionInfo
from la_solutions.dgn_elements import GetElementDescription, EXPOSE_CHILDREN_COUNT
from la_solutions.selection_set import *

# Two ways of making a smart tuple: NamedTuple is more recent in Python history
from collections import namedtuple
from typing import NamedTuple
  
from collections import namedtuple

def IsLine (elemRef: PersistentElementRef, dgnModel: DgnModel):
    """ Test whether a DGN element is a line. """
    # DGN element handlers are documented under MSPyDgnPlatform...
    # ArcHandler, AssocRegionCellHeaderHandler, BSplineCurveHandler, BSplineSurfaceHandler, BrepCellHeaderHandler, ChainHeaderHandler, 
    # ComplexHeaderDisplayHandler, ComplexShapeHandler, ComplexStringHandler, ConeHandler, CurveHandler, DgnLinkHandler, DgnLinkSetHandler,
    # DgnStoreHdrHandler, DigitalSignatureCellHeaderHandler, DimensionHandler, DisplayFilterHandler, DisplayHandler, ECInstanceHolderHandler, 
    # EllipseHandler, EllipticArcBaseHandler, ExtendedElementHandler, ExtendedNonGraphicsHandler, GroupedHoleHandler, HttpHandler, 
    # IAnnotationHandler, IRasterQuery (Raster Frame Handler), IDgnComponentDefinitionHandler, INamedViewElementHandler, 
    # IParameterExpressionHandler, ITransactionHandler, IXDataNodeHandler, IconElementHandler, LineHandler, LineStringHandler, 
    # MeshHeaderHandler, MultilineHandler, NormalCellHeaderHandler, NoteCellHeaderHandler, OleCellHeaderHandler, ParametricCellDefHandler,
    # ParametricCellHandler, PointCloudHandler, PointStringHandler, SectionClipElementHandler, SharedCellDefHandler, SharedCellHandler, 
    # SolidHandler, ShapeHandler, SurfaceHandler, SurfaceOrSolidHandler, TagElementHandler, TagSetHandler,
    # TextElemHandler, TextNodeHandler, TextTableHandler, Type2Handler

    # Create an element handle and get its Handler
    eh = ElementHandle (elemRef, dgnModel)            
    handler = eh.GetHandler() 
    # Compare this handler LineHandler class
    if isinstance (handler, LineHandler):
        # Found a line
        return True
    #else
    return False
    
def IsZeroLengthLine (elemRef: PersistentElementRef, dgnModel: DgnModel, epsilon: float = 1e-8):
    """ Test whether a DGN element is a zero-length line.
    In this example we want elements of a specified type: a DGN line element of zero-length."""

    # Create an element handle and get its Handler
    eh = ElementHandle (elemRef, dgnModel)            
    handler = eh.GetHandler() 
    # Compare this handler LineHandler class
    if isinstance (handler, LineHandler):
        # Found a line: test for zero-length
        curve = ICurvePathQuery.ElementToCurveVector (eh)
        lineLength = curve.Length()
        msg = f"Line [{elemRef.GetElementId()}] length {lineLength}"
        MessageCenter.ShowDebugMessage (msg, msg, False)
        return lineLength < epsilon
    #else
    return False

def GetLineElements(dgnModel: DgnModel = ISessionMgr.ActiveDgnModelRef.GetDgnModel())->list():
    ''' Search the specified DGN model for line elements that satisfy the test IsZeroLengthLine(). 
    Creates a list of PersistentElementRef. '''
    lines = [elemRef for elemRef in dgnModel.GetGraphicElements() if IsLine(elemRef, dgnModel)]
    nLineElements = len(lines)
    MessageCenter.ShowDebugMessage(f"Found {nLineElements} zero-length line elements", 
        f"DgnElementCounter found {nLineElements} line elements in model '{dgnModel.GetModelName()}'", False)
    return lines

def GetZeroLengthLineElements(dgnModel: DgnModel = ISessionMgr.ActiveDgnModelRef.GetDgnModel())->list():
    ''' Search the specified DGN model for line elements that satisfy the test IsZeroLengthLine(). 
    Creates a list of PersistentElementRef. '''
    lines = [elemRef for elemRef in dgnModel.GetGraphicElements() if IsZeroLengthLine(elemRef, dgnModel)]
    nLineElements = len(lines)
    MessageCenter.ShowDebugMessage(f"Found {nLineElements} zero-length line elements", 
        f"DgnElementCounter found {nLineElements} line elements in model '{dgnModel.GetModelName()}'", False)
    return lines

def Version ():
    ''' Return the version of this app. '''
    return VersionInfo("Zero Length Lines", 25, 3, 5, "Find zero-length lines")
    
if __name__ == "__main__":  # check if this script is being run directly (not imported as a module)
    vinfo = Version()
    MessageCenter.ShowDebugMessage(vinfo.brief, vinfo.verbose, False)
 