# -*- coding: utf-8 -*-

'''
Created 28-Apr-2025
See VersionInfo below in __main__ for current update
'''
from typing import List
from version_info import VersionInfo
from MSPyBentley import  WString
from MSPyDgnPlatform import  ConfigurationManager, ConfigurationVariableLevel, BentleyStatus
from MSPyMstnPlatform import MessageCenter

PATH_SEPARATOR = ";"

def get_configuration_variable_value (cfg_var_name: str, var_level: ConfigurationVariableLevel = ConfigurationVariableLevel.eUser)->(bool, str):
    ''' Get a configuration variable value as a Python string. 
    
    Wrapper around ConfigurationManager.GetVariable() which gets a non-Pythonic WString. 
    Returns: tuple(bool, str)
    '''
    status: BentleyStatus = 0
    # GetVariable obtains a Bentley WString, which is not Pythonic: it is not iterable.  
    # Convert it immediately to a Python string.
    if ConfigurationManager.IsVariableDefined(cfg_var_name):
        temp_value = WString()
        status = ConfigurationManager.GetVariable(temp_value, cfg_var_name, var_level)
        variable_value = str(temp_value)
        return (status == BentleyStatus.eSUCCESS, variable_value)
    else:
        msg = f"Configuration variable '{cfg_var_name}' is not defined"
        MessageCenter.ShowErrorMessage(msg, msg, False)
    return (False, "")
    
def evaluate_configuration_variable (cfg_var_name: str)->List[str]:
    ''' Attempt to get the value of a MicroStation configuration variable.
    
    Test whether the named variable exists, if successful get its value.
    Returns: a list of string values.
    If the variable is a multi-path then the list contains a member for each path;
    if the variable points to a single path then the list contains a single member.
    If the list is empty then the variable name is invalid or the variable doesn't exist.'''
    succeeded = False
    variable_value = str()
    (succeeded, variable_value) = get_configuration_variable_value(cfg_var_name)
    msg = str()
    variable_values = list()
    if succeeded:
        msg = f"'{cfg_var_name}' = '{variable_value}'"
        MessageCenter.ShowInfoMessage(msg, msg, False)
        variable_values = variable_value.split(PATH_SEPARATOR)
    else:
        msg = f"Unable to get value of configuration variable '{cfg_var_name}'"
        MessageCenter.ShowErrorMessage(msg, msg, False)
    return variable_values
 
def is_multi_path (path_variable_value: str)->bool:
    ''' Determine whether a variable value contains a list of paths.
    e.g. path1;path2; ...
    
    The MicroStation path separator in a configuration variable value is the semi-colon. '''    
    return PATH_SEPARATOR in path_variable_value

def trace_paths(var_name: str, path_variable_values: List[str]):
    ''' Show the contents of a configuration variable value to the message center.'''
    msg = str()
    msg = f"'{var_name}' specifies {len(path_variable_values)} paths"
    MessageCenter.ShowDebugMessage(msg, msg, False)
    for count, p in enumerate(path_variable_values):
        msg = f"path [{count}] '{p}'"
        MessageCenter.ShowDebugMessage(msg, msg, False)
    
if __name__ == "__main__":  # check if this script is being run directly (not imported as a module)
    vinfo = VersionInfo("Configuration Manager Example", 2025, 5, 1, "Evaluate a single or multi-path variable")
    MessageCenter.ShowDebugMessage(vinfo.brief, vinfo.verbose, False)
    msg = str()
    CFGVAR_MS_CELL = "MS_CELL"
    var_values = evaluate_configuration_variable(CFGVAR_MS_CELL)
    if var_values:
        # Function succeeded
        msg = "evaluate_configuration_variable SUCCESS"
        MessageCenter.ShowDebugMessage(msg, msg, False)
        trace_paths(CFGVAR_MS_CELL, var_values)
    else:
        # Function failed
        msg = "evaluate_configuration_variable failed"
        MessageCenter.ShowErrorMessage(msg, msg, False)
        
    CFGVAR_MS_TABLE_INPUT = "MS_TABLE_INPUT"
    var_values = evaluate_configuration_variable(CFGVAR_MS_TABLE_INPUT)
    if var_values:
        # Function succeeded
        msg = "evaluate_configuration_variable SUCCESS"
        MessageCenter.ShowDebugMessage(msg, msg, False)
        trace_paths(CFGVAR_MS_TABLE_INPUT, var_values)
    else:
        # Function failed
        msg = "evaluate_configuration_variable failed"
        MessageCenter.ShowErrorMessage(msg, msg, False)
        
    CFGVAR_UNKNOWN = "non-existent"
    var_value = evaluate_configuration_variable(CFGVAR_UNKNOWN)
    if var_value:
        # Function succeeded
        pass
    else:
        # Function failed
        pass
        
        
