# -*- coding: utf-8 -*-

'''
Created 26-Jan-2025
Updated 04-Mar-2025
Utility functions that operate on DGN objects.
'''

from MSPyBentley import *
from MSPyBentleyGeom import *
from MSPyDgnPlatform import *
from MSPyMstnPlatform import *

# See: https://bentleysystems.service-now.com/community?id=community_question&sys_id=3950e8461b6612147171ff7e034bcb58
# Used with ChildElemIter
EXPOSE_CHILDREN_COUNT = ExposeChildrenReason(100)
EXPOSE_CHILDREN_QUERY = ExposeChildrenReason(200)
EXPOSE_CHILDREN_EDIT  = ExposeChildrenReason(300)

ELEM_HANDLER_GET_DESCR_MAX_LENGTH = 32      
def GetElementDescription (element, stringLength: int = ELEM_HANDLER_GET_DESCR_MAX_LENGTH, dgnModel: DgnModel = ISessionMgr.ActiveDgnModelRef.GetDgnModel())->str:
    '''
    Get the human-readable description of an element from an elementRef or ElementHandle.
    '''
    description = WString()
    if isinstance (element, ElementHandle):
        handler = element.GetHandler()
        if handler is None:
            MessageCenter.ShowErrorMessage(f"GetElementDescription() ElementHandle is None!", None, False)
        else:
            handler.GetDescription (element, description, stringLength)
        
    elif isinstance (element, PersistentElementRef):
        eh = ElementHandle (element, dgnModel)
        handler = eh.GetHandler()       
        handler.GetDescription (eh, description, stringLength)
        
    return str(description)

def GetElementById(elementId: int, dgnModel: DgnModel = ISessionMgr.ActiveDgnModelRef.GetDgnModel())->[bool, ElementHandle]:
    '''
    Get an element by its ID in the supplied DGN model.
    Returns: tuple (boolean: valid, Element Handle)
    '''
    eh = ElementHandle(elementId, dgnModel)  
    valid = eh.IsValid()
    if not valid:
        msg = f"Element ID {elementId} is invalid"
        MessageCenter.ShowErrorMessage(msg, msg, False)
    
    return (valid, eh)

def GetUorsPerMeter (dgnModelRef: DgnModelRef = ISessionMgr.ActiveDgnModelRef)->float:
    '''Get the ratio of units-of-resolution per meter for specified model.
    If you don't specify a model then the active model is assumed.'''
    dgnModel = dgnModelRef.GetDgnModel()
    modelInfo = dgnModel.GetModelInfo()
    # Convert MicroStation units-of-resolution to meters
    uors: float = modelInfo.GetUorPerMeter()
    return uors

def GetUorsPerMaster (dgnModelRef: DgnModelRef = ISessionMgr.ActiveDgnModelRef)->float:
    '''Get the ratio of units-of-resolution per master unit for specified model.
    If you don't specify a model then the active model is assumed.'''
    # Convert MicroStation units-of-resolution to master units
    uors: float = ModelRef.GetUorPerMaster(dgnModelRef)
    return uors

def VerifyElementType (eh: ElementHandle, handlers: tuple)->bool:
    ''' Check that an element's class, represented by its element handler, 
    matches at least one type in the handlers tuple '''
    handler = eh.GetHandler()    
    return isinstance(handler, handlers)

def IsInvisible (eh: ElementHandle)->bool:
    ''' Determine whether a DGN element is visible. '''
    return eh.GetElement().hdr.dhdr.props.b.invisible
    
def IsVisible (eh: ElementHandle)->bool:
    ''' Determine whether a DGN element is visible. '''
    return not IsInvisible(eh)
    
def IsGraphical (eh: ElementHandle)->bool:
    ''' Determine whether a DGN element is graphical. '''
    return eh.GetElement().ehdr.isGraphics
    
def IsDeleted (eh: ElementHandle)->bool:
    ''' Determine whether a DGN element is deleted. '''
    return eh.GetElement().ehdr.deleted
    
def GetElementClass (eh: ElementHandle)->DgnElementClass:
    ''' Return the DGN class (i.e. primary, construction etc) of a DGN element. '''
    return eh.GetElement().hdr.dhdr.props.b.elementClass
    
def IsPrimaryClass (eh: ElementHandle)->bool: 
    ''' Determine whether a DGN element is primary DGN class. '''
    return GetElementClass(eh) == DgnElementClass.ePrimary

def IsConstructionClass (eh: ElementHandle)->bool: 
    ''' Determine whether a DGN element is primary DGN class. '''
    return GetElementClass(eh) == DgnElementClass.eConstruction    
    
def IsVisibleGraphicalElement (eh: ElementHandle)->bool:
    ''' Determine that this is a visible graphical DGN element. '''
    return IsGraphical(eh) and IsVisible(eh)