# -*- coding: utf-8 -*-

'''
Created 12-Mar-2025
This module wraps the Python LoadCommandTable in class CommandTableHandler.
It provides diagnostics if the XML command table (commands.xml) has errors.
'''
from os import *
import datetime
from pathlib import Path
from xml.dom import minidom
from xml.etree import ElementTree as et

from MSPyBentley import *
from MSPyMstnPlatform import *
from la_solutions.file_utilities import *
from la_solutions.version_info import VersionInfo

from enum import IntEnum
class CommandTableStatus(IntEnum):
    ''' LoadCommandTableFromXml status codes are not documented.  Codes shown here are unconfirmed by Bentley Systems. '''
    CT_SUCCESS                          = 0
    CT_ERROR                            = 32768
    CT_RESOURCENOTFOUND                 = 0x5000 + 1
    CT_BADRESOURCETYPE                  = 0x5000 + 2
    CT_BADRESOURCE                      = 0x5000 + 3
    CT_EXCEEDSMAXIMUMNESTLEVEL          = 0x5000 + 4
    CT_XMLMISSINGROOTTABLE              = 0x5000 + 0x20
    CT_XMLDUPLICATEROOTTABLE            = 0x5000 + 0x21
    CT_XMLMISSINGCOMMANDWORD            = 0x5000 + 0x22
    CT_XMLMISSINGSUBTABLE               = 0x5000 + 0x23
    CT_XMLDUPLICATESUBTABLE             = 0x5000 + 0x24
    CT_XMLBADFEATUREASPECT              = 0x5000 + 0x25
    CT_XMLDUPLICATEKEYINHANDLERSNODE    = 0x5000 + 0x26
    CT_XMLMISSINGKEYINNODE              = 0x5000 + 0x27
    CT_XMLMISSINGFUNCTIONNODE           = 0x5000 + 0x28

    CT_NOCOMMANDMATCH                   = (-1)
    CT_AMBIGUOUSMATCH                   = (-2)  

class CommandTableHandler ():
    ''' CommandTableHandler class handles XML command tables. ''' 
    # Change the COMMAND_FILE name if yours is not 'commands.xml' when you call LoadCommandTable
    COMMAND_FILE = str('commands.xml')

    def __init__(self, xmlCommandTable: str = COMMAND_FILE):
        self.xmlCommandTable = xmlCommandTable
        self.xmlFilePath = None
        self.loadStatus = CommandTableStatus.CT_ERROR
    
    @property
    def XmlCommandTable(self)->str:
        return self.xmlCommandTable
        
    def ExplainStatus (self, status: CommandTableStatus)->str:
        ''' Translate a CommandTableStatus to an explanatory string. '''
        match status:
            case CommandTableStatus.CT_SUCCESS:
                #   Command table is loaded
                return str()
            case CommandTableStatus.CT_RESOURCENOTFOUND:
                return f"Resource not found in command file '{self.xmlCommandTable}'"
            case CommandTableStatus.CT_BADRESOURCETYPE:
                return f"Bad resource type in command file '{self.xmlCommandTable}'"
            case CommandTableStatus.CT_BADRESOURCE:
                return f"Bad resource in command file '{self.xmlCommandTable}'"
            case CommandTableStatus.CT_EXCEEDSMAXIMUMNESTLEVEL:
                return f"Maximum nesting level exceeded in command file '{self.xmlCommandTable}'"
            case CommandTableStatus.CT_XMLMISSINGROOTTABLE:
                return f"Missing XML root table in command file '{self.xmlCommandTable}'"
            case CommandTableStatus.CT_XMLDUPLICATEROOTTABLE:
                return f"Duplicate XML root table in command file '{self.xmlCommandTable}'"
            case CommandTableStatus.CT_XMLDUPLICATESUBTABLE:
                return f"Duplicate XML sub table in command file '{self.xmlCommandTable}'"
            case CommandTableStatus.CT_XMLMISSINGCOMMANDWORD:
                return f"Missing command word in command file '{self.xmlCommandTable}'"
            case CommandTableStatus.CT_XMLMISSINGSUBTABLE:
                return f"Missing sub table in command file '{self.xmlCommandTable}'"
            case CommandTableStatus.CT_XMLBADFEATUREASPECT:
                return f"Bad XML feature aspect in command file '{self.xmlCommandTable}'"
            case CommandTableStatus.CT_XMLDUPLICATEKEYINHANDLERSNODE:
                # Could not trigger this error
                return f"Duplicate key in handlers node while parsing command file '{self.xmlCommandTable}'"
            case CommandTableStatus.CT_XMLMISSINGKEYINNODE:
                # Could not trigger this error
                return f"Missing key in handlers node while parsing command file '{self.xmlCommandTable}'"
            case CommandTableStatus.CT_XMLMISSINGFUNCTIONNODE:
                return f"Missing function in handlers node while parsing command file '{self.xmlCommandTable}'"
            case CommandTableStatus.CT_NOCOMMANDMATCH:
                # Could not trigger this error but missing function causes exception when keying-in the command:
                # NameError: name 'cmdTest1NonExistent' is not defined
                return f"No command match while parsing command file '{self.xmlCommandTable}'"
            case CommandTableStatus.CT_AMBIGUOUSMATCH:
                return f"Ambiguous match while parsing command file '{self.xmlCommandTable}'"
            case _:
                # Return unknown error in decimal and hexadecimal
                return f"Unknown error {status} (#{status:x}) loading command file '{self.xmlCommandTable}'"

    def LoadCommandTable (self, callingFile: str, scriptPath: str, xmlFileName: str = COMMAND_FILE)->bool:
        ''' Called, usually by main(), to load the commands from your table commands.xml. '''
        self.xmlCommandTable = xmlFileName
        #msg = str(f"LoadCommandTable scriptPath '{scriptPath}'")
        #verbose = str(f"called by '{callingFile}' scriptPath '{scriptPath}' table '{xmlFileName}'")
        #MessageCenter.ShowDebugMessage(msg, verbose, False)
        
        # Next line uses Path's overloaded operator / to build a complete file name
        self.xmlFilePath = str(Path (scriptPath) / xmlFileName)
        #msg = f"xmlFilePath='{xmlFilePath}'"
        #MessageCenter.ShowDebugMessage(msg, msg, False)
        
        # Attempt to load our command table.  If it returns something other than SUCCESS, issues a diagnostic message.
        try:
            self.loadStatus = PythonKeyinManager.GetManager().LoadCommandTableFromXml (WString (callingFile), WString (self.xmlFilePath))
            if self.loadStatus == CommandTableStatus.CT_SUCCESS:
                pass
            else:
                msg = self.ExplainStatus ( self.loadStatus)    
                MessageCenter.ShowErrorMessage(msg, msg, False)
                return False
        except Exception as e:
                msg = str(e)
                MessageCenter.ShowErrorMessage(msg, msg, False)
                return False
        return True

    def ParseTable (self)->list:
        ''' Parse an XML command table and extract the defined command key-in and related function name. '''
        msg = f"Parsing command table '{self.xmlFilePath}'"
        MessageCenter.ShowDebugMessage(msg, msg, False)
        result = list()
        doc = minidom.parse(self.xmlFilePath) 
        handlers = doc.getElementsByTagName("KeyinHandler") 
        for n, handler in enumerate(handlers):
            keyinHandler = KeyinHandler(handler.getAttribute('Keyin'), handler.getAttribute('Function'))
            #msg = f"[{n}] Keyin '{keyinHandler.command}' Function {keyinHandler.functionName}"
            #MessageCenter.ShowDebugMessage(msg, msg, False)
            result.append(keyinHandler)
        #msg = f"parsed {len(result)} key-ins"
        #MessageCenter.ShowDebugMessage(msg, msg, False)
        return result

    def ComposeHtml(self, appName: str, keyIns: list):
        ''' Document the app's key-ins in an HTML file. '''
        msg = f"App Name '{appName}' has {len(keyIns)} key-ins..."
        MessageCenter.ShowDebugMessage(msg, msg, False)
        html = et.Element('html')
        body = et.Element('body')
        html.append(body)
        title = et.SubElement(body, 'h3')
        title.text = f"App Name '{appName}'"
        date = et.SubElement(body, 'p')
        FORMAT_YearMonthDay_HourMinute = "%Y %b %d at %H:%M"
        date.text = datetime.datetime.now().strftime(FORMAT_YearMonthDay_HourMinute)
        if CommandTableStatus.CT_SUCCESS == self.loadStatus:
            status = et.SubElement(body, 'p')
            status.text = "Table Loaded Sucessfully"
        else:
            status = et.SubElement(body, 'p style="color:red;"')
            status.text = "This table has problems.  "
            status.text += self.ExplainStatus(self.loadStatus)
            
        table = et.SubElement(body, 'table')
        row = et.SubElement(table, 'tr')
        col1 = et.SubElement(row, 'th')
        col1.text = "Key-In"
        col2 = et.SubElement(row, 'th')
        col2.text = "Function"
        for n, keyIn in enumerate(keyIns):
            msg = f"[{n}] Keyin '{keyIn.command}' Function {keyIn.functionName}"
            MessageCenter.ShowDebugMessage(msg, msg, False)
            row = et.SubElement(table, 'tr')
            col1 = et.SubElement(row, 'td')
            col1.text = keyIn.command
            col2 = et.SubElement(row, 'td')
            col2.text = keyIn.functionName
            
        html_file = Path (self.xmlFilePath)
        msg = f"HTML file '{str(html_file)}' replace with '{html_file.with_suffix('.htm')}' "
        MessageCenter.ShowDebugMessage(msg, msg, False)
        try:
            et.ElementTree(html).write(html_file.with_suffix('.htm'), encoding='unicode', method='html')
            msg = f"Created HTML file '{html_file.with_suffix('.htm')}'"
            MessageCenter.ShowInfoMessage(msg, msg, False)
        except:
            msg = f"Unable to create HTML file '{html_file.with_suffix('.htm')}'"
            MessageCenter.ShowErrorMessage(msg, msg, False)

from typing import NamedTuple
class KeyinHandler (NamedTuple):
    command: str
    functionName: str

def Version ():
    ''' Return the version of this module. '''
    return VersionInfo("Command Table Tester", 25, 3, 19, "Test the syntax of an XML command table")
    
if __name__ == "__main__":  # check if this script is being run directly (not imported as a module)
    
    scriptPath = str(Path (os.path.dirname(__file__)) / 'CommandTableTester')
    handler = CommandTableHandler()
    handler.LoadCommandTable(__file__, scriptPath)
    MessageCenter.ShowDebugMessage("LoadCommandTableFromXml '{handler.XmlCommandTable}'", f"{WString (__file__)} ", False)
