# -*- coding: utf-8 -*-

'''
Created 11-Dec-2024
'''

from MSPyBentley import *
from MSPyDgnPlatform import *
from MSPyMstnPlatform import *

from collections import namedtuple

def IsInteresting (elemRef: PersistentElementRef, dgnModel: DgnModel, handlers: namedtuple):
    """ Test whether a DGN element is interesting, whatever that means.
    In this example we want elements of a specified type. """
    # DGN element handlers are documented under MSPyDgnPlatform...
    # ArcHandler, AssocRegionCellHeaderHandler, BSplineCurveHandler, BSplineSurfaceHandler, BrepCellHeaderHandler, ChainHeaderHandler, 
    # ComplexHeaderDisplayHandler, ComplexShapeHandler, ComplexStringHandler, ConeHandler, CurveHandler, DgnLinkHandler, DgnLinkSetHandler,
    # DgnStoreHdrHandler, DigitalSignatureCellHeaderHandler, DimensionHandler, DisplayFilterHandler, DisplayHandler, ECInstanceHolderHandler, 
    # EllipseHandler, EllipticArcBaseHandler, ExtendedElementHandler, ExtendedNonGraphicsHandler, GroupedHoleHandler, HttpHandler, 
    # IAnnotationHandler, IRasterQuery (Raster Frame Handler), IDgnComponentDefinitionHandler, INamedViewElementHandler, 
    # IParameterExpressionHandler, ITransactionHandler, IXDataNodeHandler, IconElementHandler, LineHandler, LineStringHandler, 
    # MeshHeaderHandler, MultilineHandler, NormalCellHeaderHandler, NoteCellHeaderHandler, OleCellHeaderHandler, ParametricCellDefHandler,
    # ParametricCellHandler, PointCloudHandler, PointStringHandler, SectionClipElementHandler, SharedCellDefHandler, SharedCellHandler, 
    # SolidHandler, ShapeHandler, SurfaceHandler, SurfaceOrSolidHandler, TagElementHandler, TagSetHandler,
    # TextElemHandler, TextNodeHandler, TextTableHandler, Type2Handler

    # Create an element handle and get its Handler
    eh = ElementHandle (elemRef, dgnModel)            
    handler = eh.GetHandler() 
    # Compare this handler to a list of handler classes
    if isinstance (handler, handlers):
        return True
    #else
    return False
        
class DgnElementCounter:
    """ A class to count DGN elements in a DGN model for one or more element types.
    
    For example, count Line and LineString elements.    
    By default, collects elements from the active model.
    For each element, it compares its handler with the provided list of element handlers.    
    """
    
    def __init__(self, dgnModel: DgnModel = ISessionMgr.ActiveDgnModelRef.GetDgnModel()):             
        self._dgnModel = dgnModel
        self._elementCount = 0
        self._searchName = ""
    
    def CountElements(self, handlers: namedtuple)->int:
        ''' Pass a namedtuple of DGN element classes (e.g. LineHandler et alia). '''
        
        msg = f"Counter.Count Elements"
        self._searchName = type(handlers).__name__
        MessageCenter.ShowDebugMessage(msg, msg, False)
        # Find the element count using a Python list comprehension
        matchedElements = [elemRef for elemRef in self._dgnModel.GetGraphicElements() if IsInteresting(elemRef, self._dgnModel, handlers)]
        self._elementCount = len(matchedElements)
        return len(matchedElements)

    def CountLines(self):
        return self.CountElements(LineHandlers)
        
    def CountEllipses(self):
        return self.CountElements(EllipseHandler)
        
    @property
    def ElementCount (self):
        ''' Get the element count. '''
        return self._elementCount
        
    @property
    def DgnModelName (self):
        ''' Get the name of the DGN model that contains elements to be counted. '''
        if (self._dgnModel is None): 
            return "unknown"
        else:
            return self._dgnModel.GetModelName()

    @property
    def SearchName(self):
        ''' Get the name of the search criteria supplied in the namedtuple. '''
        return self._searchName
    
    def __str__(self):
        return f"Element count {self._elementCount} in model '{self.DgnModelName()}'"
        
    def __repr__(self):
        return 'CountElements(Handler List)'
        
# Tuples of DGN element handlers to specify search requirements.
# DGN element handlers are documented under MSPyDgnPlatform
# See: https://stackoverflow.com/questions/42385916/inheriting-from-a-namedtuple-base-class

# tuples are a list of handler classes, but they are anonymous
#LineHandlers = (LineHandler, LineStringHandler, )
#ShapeHandlers = (ShapeHandler, )
#ArcHandlers = (ArcHandler, EllipticArcBaseHandler, )
# A circle is an ellipse, having equal axes: there is no separate CircleHandler
#EllipseHandlers = (EllipseHandler, )
# A namedtuple behaves like a tuple and has a name.
 
def MakeNamedTupleFromHandlerList (name: str, handlers)->namedtuple:
    # Construct a list of arbitrary length of named classes prior to making a namedtuple  e.g. "c1 c2 c3 ..."
    nHandlers = len(handlers)

    def expand():
        for i in range(0, len(handlers)):
            yield f"c{i}"
            
    args = ""            
    for i in expand():
        args = args + f"{i} "
    
    # Create the tuple factory
    nt_factory = namedtuple (name, args)    
    return nt_factory (*handlers)
    
LineHandlers  = MakeNamedTupleFromHandlerList("line", (LineHandler, LineStringHandler, ))     
ShapeHandlers  = MakeNamedTupleFromHandlerList("shape", (ShapeHandler, ))     
EllipseHandlers  = MakeNamedTupleFromHandlerList("ellipse", (EllipseHandler, )) 
GroupedHoleHandlers = MakeNamedTupleFromHandlerList("grouped_hole", (GroupedHoleHandler, )) 
    
if __name__ == "__main__":  # check if this script is being run directly (not imported as a module)
    MessageCenter.ShowDebugMessage("DgnElementCounter", "Count lines in active model", False)
    # Construct a new DgnElementCounter
    counter = DgnElementCounter()
    #MessageCenter.ShowDebugMessage("DgnElementCounter", "Count lines in active model", False)
    nLineElements = counter.CountElements(LineHandlers)
    MessageCenter.ShowInfoMessage(f"Found {nLineElements} {counter.SearchName} elements", 
        f"DgnElementCounter found {nLineElements} line elements in model '{counter.DgnModelName}'", False)
    counter = DgnElementCounter()
    #MessageCenter.ShowDebugMessage("DgnElementCounter", "Count shapes in active model", False)
    nShapeElements = counter.CountElements(ShapeHandlers)
    MessageCenter.ShowInfoMessage(f"Found {nShapeElements} {counter.SearchName} elements",
        f"DgnElementCounter found {nShapeElements} shape elements  in model '{counter.DgnModelName}'", False)
    #MessageCenter.ShowDebugMessage("DgnElementCounter", "Count ellipses in active model", False)
    nEllipseElements = counter.CountElements(EllipseHandlers)
    MessageCenter.ShowInfoMessage(f"Found {nEllipseElements}  {counter.SearchName}  elements", 
        f"DgnElementCounter found {nEllipseElements} {counter.SearchName} elements  in model '{counter.DgnModelName}'", False)

    nGroupedHoles = counter.CountElements(GroupedHoleHandlers)
    MessageCenter.ShowInfoMessage(f"Found {nGroupedHoles}  {counter.SearchName}  elements", 
        f"DgnElementCounter found {nGroupedHoles} {counter.SearchName} elements  in model '{counter.DgnModelName}'", False)
