# -*- coding: utf-8 -*-

'''
Created 05-Feb-2025
See VersionInfo in __main__ for current update
'''
from MSPyBentley import *
from MSPyDgnPlatform import *
from MSPyECObjects import *
from MSPyMstnPlatform import *
import numpy as np
from la_solutions.version_info import VersionInfo

# Two ways of making a smart tuple: NamedTuple is more recent in Python history
from collections import namedtuple
from typing import NamedTuple

class ItemInstanceInfo (NamedTuple):
    '''
    Class stores data about an Item instance attached to a host DGN element.
    The host's element ID allows us to retrieve that element as required.
    The Item Type library name and property name provide context for the itemValue data.
    '''
    elementId: np.dtype("int64")
    itemLibraryName: str
    itemTypeName: str
    itemPropertyName: str
    itemValue: str   # Item value may be a string, integer, float or some custom type
 
    def __str__(self):
        # Override the default __str__ function to print the data in a text instance.
        # i.e. Print in a useful format for users
        return f"[{self.itemLibraryName}.{self.itemTypeName}.{self.itemPropertyName}] = '{self.itemValue}' host {str(self.elementId)} "
    
    def __repr__(self):
        # Override the default __repr__ function to return a string representation of the
        # instance in a format that can be used to recreate the object.
        # i.e. Print in a format useful to developers
        return f"ItemInstanceInfo({str(self.elementId)}, {self.itemLibraryName}, {self.itemTypeName}, {self.itemPropertyName}, {str(self.itemValue)})"
       
    def __gt__(self, other):
        # Provide a comparison > operator so that lists of this class can be sorted.
        # I chose to sort by the text content, but you might want some other criterion
        if isinstance(other, ItemInstanceInfo):
            # Compare two strings ignoring case using casefold()
            return self.str(itemValue).casefold() > other.str(itemValue).casefold()
        else:
            return NotImplemented
   
    def __eq__(self, other):
        # Provide an equality == operator.
        # I chose to compare the Element IDs, but you might want some other criterion
        if isinstance(other, ItemInstanceInfo):
            return self.elementId == other.elementId
        else:
            return NotImplemented
        
    @classmethod
    def ColumnNames (cls)->list[str]:
        # Class method supplies a list of column names used, for example, in an Excel worksheet
       return ['Element ID', 'Library', 'Type', 'Property', 'Value']
       
if __name__ == "__main__":  # check if this script is being run directly (not imported as a module)
    vinfo = VersionInfo("ItemInstanceInfo", 25, 2, 5, "Item instance data extracted from an element")
    MessageCenter.ShowDebugMessage(vinfo.brief, vinfo.verbose, False)
    # Test class using its __str__ and __repr__ magic methods
    itemInfo = ItemInstanceInfo(1234, "library", "mythical", "imaginary", '9.876')
    print(itemInfo)
    print(repr(itemInfo))
    
    
 