# -*- coding: utf-8 -*-

'''
Created 12-Mar-2025
This module wraps the Python load_command_table in class CommandTableHandler.
It provides somewhat incomplete diagnostics if the XML command table (commands.xml) has errors.
'''
from os import *
import datetime
import string
from pathlib import Path
from xml.dom import minidom
from xml.etree import ElementTree as et

from MSPyBentley import *
from MSPyMstnPlatform import *
from la_solutions.file_utilities import *
from la_solutions.version_info import VersionInfo

from enum import IntEnum
class CommandTableStatus(IntEnum):
    ''' load_command_tableFromXml status codes are not documented.  Codes shown here are unconfirmed by Bentley Systems. '''
    CT_SUCCESS                          = 0
    CT_ERROR                            = 32768
    CT_RESOURCENOTFOUND                 = 0x5000 + 1
    CT_BADRESOURCETYPE                  = 0x5000 + 2
    CT_BADRESOURCE                      = 0x5000 + 3
    CT_EXCEEDSMAXIMUMNESTLEVEL          = 0x5000 + 4
    CT_XMLMISSINGROOTTABLE              = 0x5000 + 0x20
    CT_XMLDUPLICATEROOTTABLE            = 0x5000 + 0x21
    CT_XMLMISSINGCOMMANDWORD            = 0x5000 + 0x22
    CT_XMLMISSINGSUBTABLE               = 0x5000 + 0x23
    CT_XMLDUPLICATESUBTABLE             = 0x5000 + 0x24
    CT_XMLBADFEATUREASPECT              = 0x5000 + 0x25
    CT_XMLDUPLICATEKEYINHANDLERSNODE    = 0x5000 + 0x26
    CT_XMLMISSINGKEYINNODE              = 0x5000 + 0x27
    CT_XMLMISSINGFUNCTIONNODE           = 0x5000 + 0x28

    CT_NOCOMMANDMATCH                   = (-1)
    CT_AMBIGUOUSMATCH                   = (-2)  

class CommandTableHandler ():
    ''' CommandTableHandler class handles XML command tables. ''' 
    # Change the DEFAULT_COMMAND_FILE name if yours is not 'commands.xml' when you call load_command_table
    DEFAULT_COMMAND_FILE = str('commands.xml')

    def __init__(self, xml_command_table: str = DEFAULT_COMMAND_FILE):
        self._xml_command_table = xml_command_table
        self._xml_file_path = None
        self._load_status = CommandTableStatus.CT_ERROR
    
    @property
    def xml_command_table(self)->str:
        return self._xml_command_table
        
    @classmethod
    def calculate_script_path(cls, file: str, project_dir: str)->str:
        return str(Path (os.path.dirname(file)) / project_dir)
    
    def explain_status (self, status: CommandTableStatus)->str:
        ''' Translate a CommandTableStatus to an explanatory string. '''
        match status:
            case CommandTableStatus.CT_SUCCESS:
                #   Command table is loaded
                return str()
            case CommandTableStatus.CT_RESOURCENOTFOUND:
                return f"Resource not found in command file '{self.xml_command_table}'"
            case CommandTableStatus.CT_BADRESOURCETYPE:
                return f"Bad resource type in command file '{self.xml_command_table}'"
            case CommandTableStatus.CT_BADRESOURCE:
                return f"Bad resource in command file '{self.xml_command_table}'"
            case CommandTableStatus.CT_EXCEEDSMAXIMUMNESTLEVEL:
                return f"Maximum nesting level exceeded in command file '{self.xml_command_table}'"
            case CommandTableStatus.CT_XMLMISSINGROOTTABLE:
                return f"Missing XML root table in command file '{self.xml_command_table}'"
            case CommandTableStatus.CT_XMLDUPLICATEROOTTABLE:
                return f"Duplicate XML root table in command file '{self.xml_command_table}'"
            case CommandTableStatus.CT_XMLDUPLICATESUBTABLE:
                return f"Duplicate XML sub table in command file '{self.xml_command_table}'"
            case CommandTableStatus.CT_XMLMISSINGCOMMANDWORD:
                return f"Missing command word in command file '{self.xml_command_table}'"
            case CommandTableStatus.CT_XMLMISSINGSUBTABLE:
                return f"Missing sub table in command file '{self.xml_command_table}'"
            case CommandTableStatus.CT_XMLBADFEATUREASPECT:
                return f"Bad XML feature aspect in command file '{self.xml_command_table}'"
            case CommandTableStatus.CT_XMLDUPLICATEKEYINHANDLERSNODE:
                # Could not trigger this error
                return f"Duplicate key in handlers node while parsing command file '{self.xml_command_table}'"
            case CommandTableStatus.CT_XMLMISSINGKEYINNODE:
                # Could not trigger this error
                return f"Missing key in handlers node while parsing command file '{self.xml_command_table}'"
            case CommandTableStatus.CT_XMLMISSINGFUNCTIONNODE:
                return f"Missing function in handlers node while parsing command file '{self.xml_command_table}'"
            case CommandTableStatus.CT_NOCOMMANDMATCH:
                # Could not trigger this error but missing function causes exception when keying-in the command:
                # NameError: name 'cmdTest1NonExistent' is not defined
                return f"No command match while parsing command file '{self.xml_command_table}'"
            case CommandTableStatus.CT_AMBIGUOUSMATCH:
                return f"Ambiguous match while parsing command file '{self.xml_command_table}'"
            case _:
                # Return unknown error in decimal and hexadecimal
                return f"Unknown error {status} (#{status:x}) loading command file '{self.xml_command_table}'"

    def load_command_table (self, calling_file: str, script_path: str, xml_file_name: str = DEFAULT_COMMAND_FILE)->bool:
        ''' Called, usually by your main(), to load the commands from your table commands.xml. '''
        self._xml_command_table = xml_file_name
        #msg = str(f"load_command_table script_path '{script_path}'")
        #verbose = str(f"called by '{callingFile}' script_path '{script_path}' table '{xmlFileName}'")
        #MessageCenter.ShowDebugMessage(msg, verbose, False)
        
        # Next line uses Path's overloaded operator / to build a complete file name
        self._xml_file_path = str(Path (script_path) / xml_file_name)
        #msg = f"xml_file_path='{self._xml_file_path}'"
        #MessageCenter.ShowDebugMessage(msg, msg, False)
        
        # Attempt to load our command table.  If it returns something other than SUCCESS, issues a diagnostic message.
        try:
            self._load_status = PythonKeyinManager.GetManager().LoadCommandTableFromXml (
                WString (calling_file), WString (self._xml_file_path))
            if self._load_status == CommandTableStatus.CT_SUCCESS:
                print(f"Loaded command table '{self._xml_file_path}'")
                pass
            else:
                msg = self.explain_status ( self._load_status)    
                MessageCenter.ShowErrorMessage(msg, msg, False)
                return False
        except Exception as e:
                msg = str(e)
                MessageCenter.ShowErrorMessage(msg, msg, False)
                return False
        return True

    def parse_xml_table (self)->list:
        ''' Parse an XML command table and document the defined command key-in and its related function name. '''
        msg = f"Parsing command table '{self._xml_file_path}'"
        MessageCenter.ShowDebugMessage(msg, msg, False)
        result = list()
        doc = minidom.parse(self._xml_file_path) 
        handlers = doc.getElementsByTagName("KeyinHandler") 
        for n, handler in enumerate(handlers):
            keyinHandler = KeyinHandler(handler.getAttribute('Keyin'), handler.getAttribute('Function'))
            #msg = f"[{n}] Keyin '{keyinHandler.command}' Function {keyinHandler.functionName}"
            #MessageCenter.ShowDebugMessage(msg, msg, False)
            result.append(keyinHandler)
        #msg = f"parsed {len(result)} key-ins"
        #MessageCenter.ShowDebugMessage(msg, msg, False)
        return result

    def compose_html(self, app_name: str, key_ins: list):
        ''' Document the app's key-ins in an HTML file. '''
        msg = f"App Name '{app_name}' has {len(key_ins)} key-ins..."
        MessageCenter.ShowDebugMessage(msg, msg, False)
        html = et.Element('html')
        body = et.Element('body')
        html.append(body)
        title = et.SubElement(body, 'h3')
        title.text = f"App Name '{app_name}'"
        date = et.SubElement(body, 'p')
        FORMAT_YearMonthDay_HourMinute = "%Y %b %d at %H:%M"
        date.text = datetime.datetime.now().strftime(FORMAT_YearMonthDay_HourMinute)
        if CommandTableStatus.CT_SUCCESS == self._load_status:
            status = et.SubElement(body, 'p')
            status.text = "Table Loaded Sucessfully"
        else:
            status = et.SubElement(body, 'p style="color:red;"')
            status.text = "This table has problems.  "
            status.text += self.explain_status(self._load_status)
            
        table = et.SubElement(body, 'table')
        row = et.SubElement(table, 'tr')
        col1 = et.SubElement(row, 'th')
        col1.text = "Key-In"
        col2 = et.SubElement(row, 'th')
        col2.text = "Function"
        for n, key_in in enumerate(key_ins):
            msg = f"[{n}] Keyin '{key_in.command}' Function {key_in.functionName}"
            MessageCenter.ShowDebugMessage(msg, msg, False)
            row = et.SubElement(table, 'tr')
            col1 = et.SubElement(row, 'td')
            col1.text = key_in.command
            col2 = et.SubElement(row, 'td')
            col2.text = key_in.functionName
            
        html_file = Path (self._xml_file_path)
        msg = f"HTML file '{str(html_file)}' replace with '{html_file.with_suffix('.htm')}' "
        MessageCenter.ShowDebugMessage(msg, msg, False)
        try:
            et.ElementTree(html).write(html_file.with_suffix('.htm'), encoding='unicode', method='html')
            msg = f"Created HTML file '{html_file.with_suffix('.htm')}'"
            MessageCenter.ShowInfoMessage(msg, msg, False)
        except:
            msg = f"Unable to create HTML file '{html_file.with_suffix('.htm')}'"
            MessageCenter.ShowErrorMessage(msg, msg, False)

from typing import NamedTuple
class KeyinHandler (NamedTuple):
    ''' This tuple is populated with a command key-in and its related function when parsing XML.'''
    command: str
    functionName: str

def Version ():
    ''' Return the version of this module. '''
    return VersionInfo("Command Table Tester", 25, 8, 26, "Test the syntax of an XML command table")
    
if __name__ == "__main__":  # check if this script is being run directly (not imported as a module)
    
    handler = CommandTableHandler()
    handler.load_command_table(__file__, CommandTableHandler.calculate_script_path (__file__, 'CommandTableTester'))
    MessageCenter.ShowDebugMessage("load_command_tableFromXml '{handler.xml_command_table}'", f"{WString (__file__)} ", False)
