# -*- coding: utf-8 -*-

'''
Cell Library utilities.

Created 24-Jul-2025

'''

from MSPyBentley import *
from MSPyDgnPlatform import *    
from MSPyMstnPlatform import *
    
def get_active_cell_library()->[bool, DgnFile]:
    ''' Get a reference to the active cell library.
    Returns: tuple(True, dgnFile) if there is an active cell library. '''
    lib_name = BeFileName()
    Cell.GetLibraryName(lib_name)
    if lib_name.IsEmpty():
        return (False, None)
    else:
        (err_code, lib_reference) = Cell.GetLibraryObject(lib_name.GetWCharCP(), False)
        #print(f"err_code={err_code}")
        if err_code == 0 and lib_reference != None:
            #print(lib_reference)
            return (True, lib_reference)
        else:
            return (False, None)  

def enumerate_cell_library(lib_reference)->list[str]:
    ''' Enumerate the cells in the given cell library.
    Return a list of cell names. '''
    cell_list = list()
    if lib_reference is None:
        # Empty list
        return cell_list
    model_index: ModelIndex = lib_reference.GetModelIndex()
    #print(f"model_index {model_index}")
    cell_list = [model_item.GetName() for model_item in model_index if model_item.IsCell()]        
    return cell_list

def is_cell_in_library(cell_name: str, cell_lib: DgnFile)->bool:
    ''' Get a list of cells from a cell library.  Return True if a name
        appears in that list.  
        Case-insensitve comparison. '''
    cell_list = EnumerateCellLibrary(cell_lib)
    if cell_name.upper() in (name.upper() for name in cell_names):
        print(f"Found cell '{cell_name}'")
        return True       
    else:
        print(f"'{cell_name}' not available")
    return False

def is_cell_in_active_library(cell_name: str)->bool:
    (valid, dgnFile) = GetActiveCellLibraryObject()
    if valid:
        return is_cell_in_library(cell_name, dgnFile)
    return False
            
# Main entry point to test the Cell Library app
if __name__ == "__main__":
    ''' Pass an empty file name to GetLibraryName() to get the name of the active cell library.
    Returns an empty string if there is no currect cell library. '''
    lib_name = BeFileName()
    Cell.GetLibraryName(lib_name)
    if lib_name.IsEmpty():
        print("Cell Library name is empty (no active cell library)")
    else:
        # Pass the cell library name to GetLibraryObject(), which 
        # returns a Python tuple(error_code, object_reference).
        # If the call succeeds then error_code == 0 (eSUCCESS) and
        # lib_reference is the cell library (*.cell file).
        (valid, dgnFile) = GetActiveCellLibraryObject()
        if valid:
            valid_cell_name = "Pole 01"
            is_cell_in_library(valid_cell_name, dgnFile)
        else:
            print(f"No active cell library")
