import sys
import pathlib
import PyQt5
from PyQt5 import QtCore
from PyQt5.QtCore import QTimer, Qt, QObject, pyqtSlot
from PyQt5.QtGui import QFont,  QWindow, QValidator, QDoubleValidator, QIntValidator, QIcon
from PyQt5.QtWidgets import (
    QApplication,
    QAction,
    QComboBox, 
    QGridLayout,
    QGroupBox,
    QHBoxLayout,
    QLabel,
    QLineEdit,
    QMainWindow,
    QMenu,
    QMenuBar,
    QPushButton,
    QStackedLayout,
    QVBoxLayout,
    QWidget,
)

# Two imports for testing command callback suggestion
import time
import win32gui

from MSPyBentley import *
from MSPyBentleyGeom import *
from MSPyDgnPlatform import *
from MSPyDgnView import *
from MSPyMstnPlatform import *

from la_solutions.version_info import VersionInfo
import la_solutions.cell_library as cell_utilities
import la_solutions.file_utilities  as file_utilities

import CreateCentreLine
from CreateCentreLine.command_requests import CommandRequests
from CreateCentreLine.commands import *

#
# Place centre line UI for example Python app.
#
# This demonstrates a PyQt5 UX window containing a small number of user-interface widgets.
# The window uses a base class that contains a timer whose purpose is to trigger a windows message pump.
# Registering a timer with a 0 start interval will cause the timer method to execute every time the Qt Windows event loop finishes. 
# Calling PyCadInputQueue.PythonMainLoop from the timer will allow MicroStation's input loop to execute allowing use of the 
# MicroStation UX whilst the Qt window is displayed.
#
# Unfortunately there is a conflict between commands placed in MicroStation's input queue and Windows messaging.
# Consequently, it's impossible (2025) to have both MicroStation commands and a Python UI in the same application.
#
# In this example we choose to provide a Qt UI but no MicroStation commands.

def action_toggle_marker(w: QWindow):
    ''' Set flag to annotate points when creating chords. 
    This function is called asynchronously after the Show Markers PushButton is pressed. '''
    btn = w.find_btn_mark_points()
    # Toggle the button state
    w._mark_points = not  w._mark_points
    btn.setChecked(w._mark_points)
    if w._mark_points:
        btn.setStyleSheet("background-color : lightblue")
        btn.setIcon(w.get_named_icon('check-mark-svgrepo-com.svg'))
    else:
        btn.setStyleSheet("background-color : lightgrey")
        btn.setIcon(QIcon())

class Window(QMainWindow):
    ''' Our window inherits from QMainWindow. '''
    def __init__(self, vinfo: VersionInfo, root_dir: str, project_dir: str):
        super().__init__()
        self._root_dir = root_dir
        self._project_dir = project_dir
        self._storedWinId = self.winId()
        self._loop = QtCore.QEventLoop()
        self._versionInfo = vinfo
        self._mark_points = True
        self._picked_element_id1 = 0
        self._picked_element_id2 = 0
        self._BUTTON_TOP = 10
        self._BUTTON_LEFT = 10
        self._BUTTON_WIDTH  = 180
        self._BUTTON_HEIGHT = 100
        #   Cmd request flags
        self._cmd_pick_line1_requested = False
        self._cmd_pick_line2_requested = False
        self._cmd_execute_requested = False
        self._cmd_file_exit_requested = False
        self._cmd_help_about_requested = False
        self._cmd_help_doc_requested = False
        self._toggle_mark_points_requested = False
        #   Cmd modules
        self._MODULE_COMMANDS       = '.commands'
        p = pathlib.Path(__file__)
        self._MODULE_SELF           = f".{p.stem}"
        #   Cmd functions
        self._CMD_PICK_LINE1        =  'cmdPickLine1'
        self._CMD_PICK_LINE2        =  'cmdPickLine2'
        self._CMD_EXECUTE           =  'cmdExecute'
        self._CMD_FILE_EXIT         =  'cmdFileExit'
        self._CMD_HELP_ABOUT        =  'cmdHelpAbout'
        self._CMD_HELP_DOC          =  'cmdHelpDoc'
        self._ACTION_TOGGLE_MARKER  = 'action_toggle_marker'
        
        # Widget IDs
        self._LBL_ELEMENT_ID1       = 'lblElementId1'
        self._LBL_ELEMENT_ID2       = 'lblElementId2'
        self._LBL_POINT_COUNT       = 'lblPointCount'
        self._TXT_INTERVAL          = 'txtInterval'        
        self._BTN_MARK_POINTS       = "btnMarkPoints"
        
        self.initUI()
        # Give the window a unique name for easier identification if needed for cleanup.
        self.setObjectName("dlgPlaceCentreLine")
        # A list of command requests for this AddIn.  A request is typically a function call, rather than a MicroStation key-in command
        requests = {(self._MODULE_COMMANDS, 
                    self._CMD_PICK_LINE1): False,
                    (self._MODULE_COMMANDS, 
                    self._CMD_PICK_LINE2): False,
                    (self._MODULE_COMMANDS, 
                    self._CMD_EXECUTE): False,
                    (self._MODULE_COMMANDS, 
                    self._CMD_FILE_EXIT): False,
                    (self._MODULE_COMMANDS, 
                    self._CMD_HELP_ABOUT): False,
                    (self._MODULE_COMMANDS, 
                    self._CMD_HELP_DOC): False,
                    (self._MODULE_SELF,
                    self._ACTION_TOGGLE_MARKER): False}
        self._requests = CommandRequests(requests)
        print(self._requests)

    @pyqtSlot()
    def queue_action_pick_line1(self):
        ''' Start the cmd_pick_line state engine. '''
        msg = "queue_action_pick_line1"
        MessageCenter.ShowDebugMessage(msg, msg, False) 
        print(msg)
        self._cmd_pick_line1_requested = True
     
    @pyqtSlot()
    def queue_action_pick_line2(self):
        ''' Start the cmd_pick_line state engine. '''
        msg = "queue_action_pick_line2"
        MessageCenter.ShowDebugMessage(msg, msg, False) 
        print(msg)
        self._cmd_pick_line2_requested = True
     
    @property
    def picked_element_id1(self)->int:
        return self._picked_element_id1
        
    @property
    def picked_element_id2(self)->int:
        return self._picked_element_id2
        
    @property
    def interval(self)->int:
        txt: QLineEdit = self.find_txt_interval()
        content = txt.text()
        return int(content)
        
    @pyqtSlot()
    def queue_action_execute(self):
        ''' Send a key-in string to MicroStation. '''
        msg = "queue_action_execute"
        MessageCenter.ShowDebugMessage(msg, msg, False) 
        self._cmd_execute_requested = True

    @pyqtSlot()
    def action_toggle_mark_points(self):
        self._toggle_mark_points_requested = True
        print("action_toggle_mark_points")

    @pyqtSlot()
    def queue_action_file_exit(self):
        ''' Set a flag to request an action. '''
        msg = "queue_action_execute"
        MessageCenter.ShowDebugMessage(msg, msg, False) 
        self._cmd_file_exit_requested = True
       
    def makeLabel(self)->QLabel:
        ''' Create a label widget. '''
        label = QLabel(self)
        #   Labels can understand HTML markup
        label.setText(f"<h3>{self._versionInfo.verbose}</h3>")
        return label

    def help_about_call(self):
        ''' Open a web browser tab.'''
        print("help_about_call()")
        self._cmd_help_about_requested = True

    def help_doc_call(self):
        ''' Open a web browser tab.'''
        print("help_doc_call()")
        self._cmd_help_doc_requested = True

    def exit_call(self):
        MessageCenter.ShowDebugMessage("Exit", "Exit this app", False)
        #self.close()  
        self._cmd_file_exit_requested = True
        
    def make_action_exit (self):
        action = QAction(self.get_named_icon('exit-svgrepo-com.svg'), '&Exit', self)
        action.setShortcut('Ctrl+Q')
        action.setStatusTip('Exit application')
        action.triggered.connect(self.exit_call)
        return action

    def make_action_help_about (self):
        action = QAction(self.get_named_icon('icons8-info-64.svg'), '&About', self)        
        action.setShortcut('')
        action.setStatusTip('Help About')
        action.triggered.connect(self.help_about_call)     
        return action
        
    def make_action_help_doc (self):
        action = QAction(self.get_named_icon('book-svgrepo-com.svg'), '&Documentation', self)        
        action.setShortcut('')
        action.setStatusTip('Help Documentation')
        action.triggered.connect(self.help_doc_call)     
        return action
        
    def make_menu_actions (self, file_menu, help_menu):
        ''' A Qt menu requires an action.  When you invoke the action by choosing a menu item,
        Python reacts by calling the target of that action through the QAction.triggered.connect(target) call.

        In this small app, the action of File|Exit is the exit_call() function, and the 
        action of Help|Documentation is the help_doc_call() function. '''         
        # Create menu actions
        file_menu.addAction(self.make_action_exit())
        help_menu.addAction(self.make_action_help_about())
        help_menu.addAction(self.make_action_help_doc())
        
    def make_menus(self):
        ''' How to right-align a menu...
        https://stackoverflow.com/questions/8726677/aligning-qmenubar-items-add-some-on-left-and-some-on-right-side '''
        menuBar = self.menuBar()
        file_menu = menuBar.addMenu('&File')
        help_menu = menuBar.addMenu('&Help')
        
        right_menu_bar = QMenuBar(menuBar)
        right_menu = QMenu("Test", right_menu_bar)
        menuBar.setCornerWidget(right_menu_bar)
        #menuBar.setCornerWidget(help_menu)
        # Add an action for each menu
        self.make_menu_actions (file_menu, help_menu)

    def makePointCountLabel(self)->QLabel:
        ''' Create a label widget. '''
        label = QLabel(self)
        #   Labels can understand HTML markup
        label.setText("Points Created: ")
        label.setObjectName(self._LBL_POINT_COUNT)
        return label

    def makePickButton1(self)->QPushButton:
        ''' Create a push button to start the Pick command. '''
        btn = QPushButton(f"Pick Line 1")
        btn.setFixedWidth(self._BUTTON_WIDTH)
        btn.clicked.connect(self.queue_action_pick_line1 )
        # Not allowed
        # btn.clicked(PyCadInputQueue.SendKeyin())
        return btn
    
    def makePickButton2(self)->QPushButton:
        ''' Create a push button to start the Pick command. '''
        btn = QPushButton(f"Pick Line 2")
        btn.setFixedWidth(self._BUTTON_WIDTH)
        btn.clicked.connect(self.queue_action_pick_line2 )
        # Not allowed
        # btn.clicked(PyCadInputQueue.SendKeyin())
        return btn
    
    def makeExecuteButton(self)->QPushButton:
        ''' Create a push button to start the Pick command. '''
        btn = QPushButton("Execute")
        btn.setFixedWidth(self._BUTTON_WIDTH)
        btn.clicked.connect(self.queue_action_execute)
        # Not allowed
        # btn.clicked(PyCadInputQueue.SendKeyin())
        return btn
    
    def makeExecuteButton(self)->QPushButton:
        ''' Create a push button to start the Pick command. '''
        btn = QPushButton("Execute")
        btn.setFixedWidth(self._BUTTON_WIDTH)
        btn.clicked.connect(self.queue_action_execute)
        # Not allowed
        # btn.clicked(PyCadInputQueue.SendKeyin())
        return btn
        
    def makeMarkPointsButton (self)->QPushButton:
        btn = QPushButton(self.get_named_icon('check-mark-svgrepo-com.svg'), "", None)
        btn.setFixedWidth(int(self._BUTTON_WIDTH / 4))
        btn.setCheckable(True)
        btn.setObjectName(self._BTN_MARK_POINTS)
        btn.clicked.connect(self.action_toggle_mark_points)
        return btn

    @property
    def mark_points(self)->bool:
        ''' Returns the state of the Mark Points variable. '''
        return self._mark_points
        
    def get_named_icon(self, icon_name: str)->QIcon:
        ''' Get an icon from our project's Icons folder. '''
        project_dir = Path (self._root_dir) / self._project_dir
        return QIcon(f"{file_utilities.get_icon_from_root_sub_folder(project_dir, icon_name)}")
        
    def makeVersionLabel (self, vinfo: VersionInfo)->QLabel:           
        label = QLabel(self, alignment=Qt.AlignmentFlag.AlignRight | Qt.AlignmentFlag.AlignBottom)
        label.setText(vinfo.brief)
        return label      
        
    def makeIntervalLabel (self, text)->QLabel:           
        label = QLabel(self, alignment=Qt.AlignmentFlag.AlignRight | Qt.AlignmentFlag.AlignBottom)
        label.setText(text)
        return label      
        
    def makeMarkPointsLabel (self, text)->QLabel:           
        label = QLabel(self, alignment=Qt.AlignmentFlag.AlignRight | Qt.AlignmentFlag.AlignBottom)
        label.setText(text)
        return label      
    
    def makeIntervalText(self)->QLineEdit:
        txt =  QLineEdit(self)
        txt.setFixedWidth(self._BUTTON_WIDTH)
        # Check that user enters a valid number for the interval between cells.
        # In this example, the interval must lie in the range 1..100
        _MIN_VAL = 1
        _MAX_VAL = 100
        _INITIAL_INTERVAL = 1   # Master Units
        txt.setPlaceholderText("Interval must be a positive number 1..100")
        txt.setValidator(QIntValidator(_MIN_VAL, _MAX_VAL, txt))
        txt.setObjectName(self._TXT_INTERVAL)
        txt.setText(str(_INITIAL_INTERVAL))
        return txt
        
    def makeLaSolutionsLabel (self)->QLabel:
        label = QLabel(self, alignment=Qt.AlignmentFlag.AlignBottom)
        label.setText("<h3>LA Solutions</h3>")
        # Display the label using a style-sheet to obtain CSS font attributes
        label.setStyleSheet('color: teal; font-family: Times New Roman; font-size: 16pt')
        return label

    def makePickedIdLabel1 (self, text)->QLabel:           
        label = QLabel(self, alignment = Qt.AlignmentFlag.AlignRight | Qt.AlignmentFlag.AlignBottom)
        label.setObjectName(self._LBL_ELEMENT_ID1)
        label.setText(text)
        return label      

    def makePickedIdLabel2 (self, text)->QLabel:           
        label = QLabel(self, alignment = Qt.AlignmentFlag.AlignRight | Qt.AlignmentFlag.AlignBottom)
        label.setObjectName(self._LBL_ELEMENT_ID2)
        label.setText(text)
        return label      

    def find_named_widget(self, name)->QWidget: 
        '''
        Search children of this QWindow for the named QWidget.
        '''
        widget: QWidget = self.findChild(QWidget, name)
        if widget == None:
            print(f"Widget '{name}' not found")
        return widget
    
    def find_lbl_point_count(self)->QLabel:
        '''
        Return the named QLabel.
        '''
        return self.find_named_widget(self._LBL_POINT_COUNT)

    def set_lbl_point_count(self, n: int):
        lbl = self.find_lbl_point_count()
        lbl.setText(f"Point Count: {n}")
        
    def find_lbl_element_id1(self)->QLabel:
        '''
        Return the named QLabel.
        '''
        return self.find_named_widget(self._LBL_ELEMENT_ID1)
    
    def find_lbl_element_id2(self)->QLabel:
        '''
        Return the named QLabel.
        '''
        return self.find_named_widget(self._LBL_ELEMENT_ID2)
    
    def find_txt_interval(self)->QLineEdit:
        '''
        Return the named QLineEdit.
        '''
        return self.find_named_widget(self._TXT_INTERVAL)
    
    def find_btn_mark_points(self)->QPushButton:
        '''
        Return the named QPushButton.
        '''
        return self.find_named_widget(self._BTN_MARK_POINTS)
    
    def is_duplicate_id (self, id: int)->bool:
        if id == self._picked_element_id1:
            return True
        if id == self._picked_element_id2:
            return True
        return False
    
    def clear_picked_element_ids(self):
            self._picked_element_id1  = 0
            lbl = self.find_lbl_element_id1()
            lbl.setText(str(self._picked_element_id1))    
            self._picked_element_id2  = 0
            lbl = self.find_lbl_element_id2()
            lbl.setText(str(self._picked_element_id2))    
            
    def set_picked_element_id1(self, id: int):
        '''
        Lets an external module set our stored Element ID.
        On successful receipt of the value, update the label Element ID.
        '''
        if id > 0:
            self._picked_element_id1  = id
            lbl = self.find_lbl_element_id1()
            lbl.setText(str(self._picked_element_id1))
        else:
            msg = "set_picked_element_id: Picked element ID must be a positive integer"
            print(msg)
               
    def set_picked_element_id2(self, id: int):
        '''
        Lets an external module set our stored Element ID.
        On successful receipt of the value, update the label Element ID.
        '''
        if id > 0:
            self._picked_element_id2  = id
            lbl = self.find_lbl_element_id2()
            lbl.setText(str(self._picked_element_id2))
        else:
            msg = "set_picked_element_id: Picked element ID must be a positive integer"
            print(msg)
      
    @property
    def picked_element_id1(self)->int:
        return self._picked_element_id1 
      
    @property
    def picked_element_id2(self)->int:
        return self._picked_element_id2
        
    def _set_requested_flags(self):
        ''' This rather clunky logic is used to avoid problems with the QWidget actions interfering with MicroStation Python function calls.'''
        #print("_set_requested_flags")
        if self._cmd_pick_line1_requested:
            print("_cmd_pick_line1_requested")
            self._requests.set_requested(self._CMD_PICK_LINE1)
            self._cmd_pick_line1_requested = False
        if self._cmd_pick_line2_requested:
            print("_cmd_pick_line2_requested")
            self._requests.set_requested(self._CMD_PICK_LINE2)
            self._cmd_pick_line2_requested = False
        if self._cmd_execute_requested:
            print("_cmd_execute_requested")
            self._requests.set_requested(self._CMD_EXECUTE)
            self._cmd_execute_requested = False
        if self._cmd_file_exit_requested:
            print("_cmd_file_exit_requested")
            self._requests.set_requested(self._CMD_FILE_EXIT)
            self._cmd_file_exit_requested = False
        if self._cmd_help_about_requested:
            print("_cmd_help_about_requested")
            self._requests.set_requested(self._CMD_HELP_ABOUT)
            self._cmd_help_about_requested = False
        if self._cmd_help_doc_requested:
            print("_cmd_help_doc_requested")
            self._requests.set_requested(self._CMD_HELP_DOC)
            self._cmd_help_doc_requested = False
        if self._toggle_mark_points_requested:
            self._requests.set_requested(self._ACTION_TOGGLE_MARKER)
            self._toggle_mark_points_requested = False
        
    def ms_mainLoop(self):
        """
        Custom main loop that pumps both Qt and MicroStation events,
        and executes any deferred tool-start calls in a safe context.
        This loop is the key to letting the UI and tool coexist.
        """
        #while self.isVisible():
        while win32gui.IsWindow(self._storedWinId):
            # Standard pump for a hybrid Qt/MicroStation application
            #QApplication.instance().processEvents()
            self._loop.processEvents()
            PyCadInputQueue.PythonMainLoop()
            #print(f"requests: {self._requests}")
            self._set_requested_flags()
                
            # Check if a tool start was requested by a QWidget action
            ((mod_name, func), flag) = self._requests.have_request()
            if flag:
                # If yes, run the stored function (e.g., InstallNewInstance)
                self._requests.invoke_requested(self)
                # Clear the flag so it only runs once per click
                self._requests.clear_requests()
                
            time.sleep(0.001)

    def initUI(self):
        ''' Initialise the PyQt user interface. '''
        self.setWindowTitle(self._versionInfo.brief)
        self.setGeometry(100, 200, 720, 320)
        # Create menu bar and add actions
        self.make_menus ()

        outer_layout = QGridLayout()
        inner_layout = QGridLayout()
        # Row 0
        outer_layout.addWidget(self.makeLabel(), 0, 0)
        inner_layout.addWidget(self.makePickButton1(), 0, 1)
        inner_layout.addWidget(self.makePickButton2(), 0, 2)
        inner_layout.addWidget(self.makeExecuteButton(), 0, 3)
        # Row 1
        inner_layout.addWidget(self.makePickedIdLabel1("Element 1 ID"), 1, 1)
        inner_layout.addWidget(self.makePickedIdLabel2("Element 2 ID"), 1, 2)
        # Row 2
        inner_layout.addWidget(self.makeIntervalLabel("Interval:"), 2, 0)
        inner_layout.addWidget(self.makeIntervalText(), 2, 1)
        inner_layout.addWidget(self.makeMarkPointsLabel("Mark Points:"), 2, 2)
        inner_layout.addWidget(self.makeMarkPointsButton(), 2, 3)
        # Row 3
        inner_layout.addWidget(self.makePointCountLabel(), 3, 1)
        
        outer_layout.addLayout(inner_layout, 1, 0)
        outer_layout.addWidget(self.makeLaSolutionsLabel(), 2, 0)
        outer_layout.addWidget(self.makeVersionLabel(self._versionInfo), 2, 1)

        widget = QWidget()
        widget.setLayout(outer_layout)
        self.setCentralWidget(widget)
                    
    @classmethod
    def Run(cls, vinfo: VersionInfo, root_dir: str, project_dir: str):
        ''' Start PyQt and the main processing loop. 
        
        Defined as a class method.  Call using Window.Run() from __main__. '''
        app = QApplication([])
        w = Window(vinfo, root_dir, project_dir)
        w.show()
        w.ms_mainLoop()

if __name__ == "__main__":  # check if this script is being run directly (not imported as a module)
    Run()
