# -*- coding: utf-8 -*-

'''
Created 06-Aug-2025
This is the implemention of the Pick Line class.
'''
from MSPyBentley import *
from MSPyBentleyGeom import DPoint3d, DPoint2d, ICurvePrimitive, CurveLocationDetail
from MSPyDgnPlatform import *
from MSPyMstnPlatform import *
from MSPyDgnView import *

from PyQt5.QtGui import QWindow

#from CreateCentreLine.place_centre_line_qt import Window
import CreateCentreLine.place_centre_line_qt

# Borrowed from dgn_elements
ELEM_HANDLER_GET_DESCR_MAX_LENGTH = 32      
def GetElementDescription   (element, 
                            string_length: int = ELEM_HANDLER_GET_DESCR_MAX_LENGTH, 
                            dgn_model: DgnModel = ISessionMgr.ActiveDgnModelRef.GetDgnModel())->str:
    '''
    Get the human-readable description of an element from an elementRef or ElementHandle.
    
    params:
    element [in] is an ElementHandle or ElementRef
    string_length [in] is the maximum length of the description string you want to accept; default to 32 
    dgn_model [in] is a DGN model that contains element; defaults to the active model
    '''
    description = WString()
    if isinstance (element, ElementHandle):
        handler = element.GetHandler()
        if handler is None:
            MessageCenter.ShowErrorMessage(f"GetElementDescription() ElementHandle is invalid", None, False)
        else:
            handler.GetDescription (element, description, string_length)
        
    elif isinstance (element, PersistentElementRef):
        eh = ElementHandle (element, dgn_model)
        handler = eh.GetHandler()       
        handler.GetDescription (eh, description, string_length)
        
    return str(description)

class cmdPickLineElement(DgnElementSetTool):
    '''
    Inherits from DgnElementSetTool.  Prompts user to pick a line element.
    'line element' means anything that converts to an open CurveVector.
    '''
    _ACCEPTABLE_TYPES = (ICurvePrimitive.eCURVE_PRIMITIVE_TYPE_Line,
                        ICurvePrimitive.eCURVE_PRIMITIVE_TYPE_LineString,
                        ICurvePrimitive.eCURVE_PRIMITIVE_TYPE_Arc,
                        ICurvePrimitive.eCURVE_PRIMITIVE_TYPE_CurveVector,
                        ICurvePrimitive.eCURVE_PRIMITIVE_TYPE_BsplineCurve,
                        )

    def __init__(self, select = 1, window: QWindow = None, toolId = 0):
        ''' Initialise this class and its base class DgnElementSetTool.  '''
        # Don't call super()
        DgnElementSetTool.__init__(self, toolId) # C++ base's __init__ must be called.
        self.m_isDynamics = False
        self.m_ev = None
        self.m_self = self # Keep self referenced
        self._qt_window = window # The Qt window that started this command
        self._select = select # The line we want to pick: 1 or 2 in this example
        
    def _GetToolName(self, name):
        name = WString(f"Pick line {self._select}")
        
    def _DoGroups(self):
        '''
        Pick single element, don't include graphic and named group members.
        '''
        return False 
        
    def _AllowSelection(self):
        ''' Selection set not allowed. '''
        return DgnElementSetTool.eUSES_SS_None 
        
    def _SetupForModify(self, ev, isDynamics):
        '''  '''
        self.m_isDynamics = isDynamics
        self.m_ev = ev
        
        return True
    
    def _is_acceptable_primitive_type(self, primitiveType, acceptable_types: list)->bool:
        '''
        Test this primitive type against a list of acceptable types.
        '''
        return primitiveType in acceptable_types
            
    def _OnElementModify(self, eeh)->int:
        '''
        Since we've already validated the element in OnPostLocate and don't support groups/selection sets 
        we don't have to check it again here.
        '''
        curve = ICurvePathQuery.ElementToCurveVector(eeh)
        primitive = curve[0]
        location = CurveLocationDetail()

        locatePoint = DPoint3d()
        self._GetAnchorPoint(locatePoint)

        ''' Get curve location detail from accept point. '''
        if not curve.ClosestPointBounded(locatePoint, location):
            return BentleyStatus.eERROR
       
        primitiveType = primitive.GetCurvePrimitiveType()
        #print (f"_OnElementModify primitiveType '{primitiveType}'")
        if self._qt_window and self._is_acceptable_primitive_type(primitiveType, self._ACCEPTABLE_TYPES):
            if 1 == self._select:
                self._qt_window.set_picked_element_id1(eeh.GetElementId())
            else:
                self._qt_window.set_picked_element_id2(eeh.GetElementId())
            #msg =  f"cmdPickLine picked {self._select} {GetElementDescription(eeh)} ID {eeh.GetElementId()}"
            #print(msg)            
            return BentleyStatus.eSUCCESS
        # Return ERROR to signify no change.    
        return BentleyStatus.eERROR
        
    def _OnPostLocate(self, path, cantAcceptReason)->bool:
        '''
        Called during a locate to allow tool to filter which elements can be selected.  
        '''
        #msg = "_OnPostLocate"
        #MessageCenter.ShowDebugMessage(msg, msg, False) 
       
        if not DgnElementSetTool._OnPostLocate(self, path, cantAcceptReason):
            #msg = "DgnElementSetTool._OnPostLocate (super) returned false"
            #MessageCenter.ShowErrorMessage(msg, msg, False) 
            return False

        eh = ElementHandle(path.GetHeadElem(), path.GetRoot())
        
        if self._qt_window.is_duplicate_id(eh.GetElementId()):
            cantAcceptReason = "Same element selected"
            return False
        curve = ICurvePathQuery.ElementToCurveVector(eh)

        # Accept elements that are open paths with one or more linear segments (e.g. line or linestring).
        if curve == None or (not curve.IsOpenPath()):
            # This produces a  lot of messages as you move the cursor
            cantAcceptReason = "Not an open element"
            MessageCenter.ShowErrorMessage(cantAcceptReason, cantAcceptReason, False) 
            return False

        primitiveType = curve.HasSingleCurvePrimitive()
        #print (f"_OnPostLocate: primitiveType '{primitiveType}'")
        if self._is_acceptable_primitive_type(primitiveType, self._ACCEPTABLE_TYPES):
            #print(f"_OnPostLocate accepted primitive type {primitiveType}")                                 
            return True
       
        cantAcceptReason = f"primitiveType={primitiveType} Not an open element"
        msg = cantAcceptReason
        MessageCenter.ShowErrorMessage(msg, msg, False) 
        return False
     
    def _SetupAndPromptForNextAction(self):
        '''
        An opportunity to display a message to your user
        '''
        msg = f'Pick Line or Arc {self._select}'
        NotificationManager.OutputPrompt(msg)
        
    def _OnRestartTool(self):
        '''    
        Install a new instance of the tool. Will be called in response to external events
        such as undo or by the base class from _OnReinitialize when the tool needs to be
        reset to it's initial state.
        '''
        self.InstallNewInstance(self._select, self._qt_window, 0)
    
    @classmethod
    def InstallNewInstance(self, select: int = 1, w: QWindow = None, toolId = 0):
        '''
        Method to create and install a new instance of the tool. If InstallTool returns ERROR,
        the new tool instance will be freed/invalid. 
        '''
        print(f"cmd_pick_line.InstallNewInstance({select})")
        tool = cmdPickLineElement(select, w, toolId)
        tool.InstallTool()

if __name__ == "__main__":  # check if this script is being run directly (not imported as a module)
    msg = "Implementation of state machine for Pick Line"
    MessageCenter.ShowDebugMessage(msg, msg, False)

    cmdPickLineElement.InstallNewInstance()
