# -*- coding: utf-8 -*-

'''
Created 06-Aug-25
This is the implemention of the Execute class for place centre line.
Given two element IDs, it extracts the geometry of those features and then places the
centre line at specified intervals along those linear feature.
'''
from MSPyBentley import *
from MSPyBentleyGeom import *
from MSPyDgnPlatform import *
from MSPyMstnPlatform import *
from MSPyDgnView import *

from PyQt5.QtGui import QWindow

import math

#import la_solutions.cell_library as cell_utilities

class create_centre_line():
    '''
    Given the ID of a linear element, place a cell at regular intervals along that line.
    '''
    def __init__(self, window: QWindow):
        ''' Initialise this class.  '''
        self._picked_element_id1 = window.picked_element_id1
        self._picked_element_id2 = window.picked_element_id2
        self._interval = window.interval
        # The Qt window that started this command.
        # We post results to widgets in that window.
        self._qt_window = window 
        self._qt_window.set_lbl_point_count(0)
        print(f"create_centre_line Line1 ID = {self._picked_element_id1}  Line2 ID = {self._picked_element_id2} interval={self._interval} ")

    def is_valid_element1(self):
        ''' Check that the element ID is valid.
        You may want to go further and verify that the ID is that of a linear element. '''
        return self._picked_element_id1 > 0 and self._picked_element_id1 != self._picked_element_id2
    
    def is_valid_element2(self):
        ''' Check that the element ID is valid.
        You may want to go further and verify that the ID is that of a linear element. '''
        return self._picked_element_id2 > 0 and self._picked_element_id1 != self._picked_element_id2
     
    def is_valid_interval(self):
        ''' Check that the supplied interval is valid.  It must be a positive number. '''
        return self._interval > 0.0
        
    def _requirements_met(self)->bool:
        ''' Check that user has supplied the necessary parameters to construct chords between lines.  '''
        if  (self.is_valid_element1()
            and
            self.is_valid_element2()
            and
            self.is_valid_interval()):
            return True
        return False
        
    def construct_chords_between_lines(self, line_element_id1: int, line_element_id2: int, interval: float)->int:
        """
        Place chords at regular intervals between two line elements in the active model.

        :param line_element_id1: The element ID of the first line.
        :param line_element_id2: The element ID of the second line.
        :param interval: The interval (in master units) at which to construct chords.
        :returns: Number of chords created.    

        """
        ACTIVEMODEL = ISessionMgr.ActiveDgnModelRef
        dgnModel = ACTIVEMODEL.GetDgnModel()
        modelInfo = dgnModel.GetModelInfo()
        __, uor_per_master = modelInfo.StorageUnit.ConvertDistanceFrom(modelInfo.UorPerStorage, modelInfo.MasterUnit)
       
        interval_uor = uor_per_master * interval
        
        line1_eh = ElementHandle(line_element_id1, dgnModel)
        cv1 = ICurvePathQuery.ElementToCurveVector(line1_eh)
        line2_eh = ElementHandle(line_element_id2, dgnModel)
        cv2 = ICurvePathQuery.ElementToCurveVector(line2_eh)
        # Calculate number of points to create            
        l1 = cv1.Length()
        l2 = cv2.Length()
        n1 = 1 + int(l1 / interval_uor)
        n2 = 1 + int(l2 / interval_uor)
        print (f"Line 1 length={l1 / uor_per_master} n1={n1} Line 2 length={l2 / uor_per_master} n2={n2} interval={self._interval}")
        distances = DoubleArray()
        for i in range(n1):
            distances.append(i * interval_uor)
        locations1 = CurveLocationDetailArray()
        cv1.AddSpacedPoints(distances, locations1)
        distances.clear()
        for i in range(n2):
            distances.append(i * interval_uor)
        locations2 = CurveLocationDetailArray()
        cv2.AddSpacedPoints(distances, locations2)
        locations = list(zip(locations1, locations2))
        n_locs = len(locations)
        p1 = DPoint3d()
        p2 = DPoint3d()
        text_size = .1 # Master Units
        mid_point = DPoint3d()
        mid_points = DPoint3dArray()
        for i in range(n_locs):
            detail1, detail2 = locations[i]
            p1 = self.get_point_from_location_detail(detail1)
            p2 = self.get_point_from_location_detail(detail2)
            # Draw the chord between corresponding points.
            # Use during development for visual confirmation of your calculations.
            self.create_chord (p1, p2, dgnModel)  
            if self._qt_window.mark_points:
                # Create a text marker showing the index of the chord end points  
                marker = f"M{i}"
                self.create_marker (marker, p1, dgnModel, text_size  * uor_per_master)
                marker = f"N{i}"
                self.create_marker (marker, p2, dgnModel, text_size  * uor_per_master)
            # The centre line is created from a list of mid-points.
            mid_point = self.calculate_mid_point(p1, p2)
            mid_points.append(mid_point)
        print(f"found {n_locs} CurveLocationDetails")
        self.construct_line(mid_points, dgnModel)
        
        return n_locs

    def calculate_mid_point(self, p1: DPoint3d, p2: DPoint3d)->DPoint3d:
        ''' Calculate the mid-point between two DPoin3d. '''
        # Make a copy of p1 
        mid_point = DPoint3d(p1.x, p1.y, p1.z)
        # These are in-place operations
        mid_point.Add(p2)
        mid_point.Scale(0.5)
        return mid_point

    def construct_line(self, points: DPoint3dArray, dgnModel)->bool:
        ''' Create a line-string from a point array. '''
        print("construct_line()")
        line_string = EditElementHandle()
        status = LineStringHandler.CreateLineStringElement (line_string, None, points, dgnModel.Is3d(), dgnModel)
        if eSUCCESS == status:
            ElementPropertyUtils.ApplyActiveSettings(line_string)
            print(f"Created line_string")
            status = line_string.AddToModel()
            return eSUCCESS == status
        else:
            print(f"Failed to create line_string")
        return False            
        
    def get_point_from_location_detail(self, detail: CurveLocationDetail)->DPoint3d:
        vray: ValidatedDRay3d = detail.PointAndUnitTangent()
        ray: DRay3d = vray.Value()
        p1 = DPoint3d()
        p2 = DPoint3d()
        ray.EvaluateEndPoints(p1, p2)
        return p1

    def create_text_block (self, text_size, dgnModel)->TextBlock:
        # Create a variable to add the text properties.
        textBlockProperties = TextBlockProperties.Create(dgnModel)
        # Create a variable to add the paragraph properties.
        paragraphProperties = ParagraphProperties.Create(dgnModel)
        # Create a text font.
        runPropertiesFont = DgnFontManager.GetDefaultTrueTypeFont()
        # Create a text size.
        runPropertiesSize = DPoint2d(text_size, text_size)
        # Add the text properties.
        runProperties = RunProperties.Create(runPropertiesFont, runPropertiesSize, dgnModel)
        # Textblock added the text strings to be displayed with the before-mentioned properties.
        textBlock = TextBlock(textBlockProperties, paragraphProperties, runProperties, dgnModel)
        return textBlock
        
    def create_marker (self, marker: str, p: DPoint3d, dgnModel, text_size):
        ''' Create a text element showing variable i.
        A debug helper routine to help identify generated points. '''
        textBlock = self.create_text_block(text_size, dgnModel)
        textBlock.AppendText(marker)
        textBlock.SetUserOrigin(p)
        textEditElementHandle = EditElementHandle()
        result = TextHandlerBase.CreateElement(textEditElementHandle, None, textBlock)
        if result == eTEXTBLOCK_TO_ELEMENT_RESULT_Success:
            #print(f"Created text {marker}")
            status = textEditElementHandle.AddToModel()
            if eSUCCESS == status:
                #print(f"Added text {marker}")
                pass
            else:
                print(f"Failed to add text {marker}")
            
        else:
            print(f"Failed to create text {marker}")
        
        
    def create_chord (self, p1: DPoint3d, p2: DPoint3d, dgn_model)->bool:
        ''' Construct a line between points p1 and p2 and calculate their mid-point. '''
        created = False
        segment = DSegment3d(p1.x, p1.y, p1.z, p2.x, p2.y, p2.z)
        eeh = EditElementHandle()
        status = LineHandler.CreateLineElement (eeh, None, segment, dgn_model.Is3d (), dgn_model)
        if BentleyStatus.eSUCCESS == status:
             status = eeh.AddToModel ()
        return BentleyStatus.eSUCCESS == status
        
    def execute(self)->int:
        ''' Calculate points along each of two lines at regular intervals.
        Construct chords between corresponding points.
        Calculate the mid-point of each chord.
        '''
        chord_count = 0
        if self._requirements_met():
            print ("Proceed with chord calculation")
            chord_count = self.construct_chords_between_lines(self._picked_element_id1, self._picked_element_id2, self._interval)
            self._qt_window.set_lbl_point_count (chord_count)
            self._qt_window.clear_picked_element_ids()
            PyCommandState.StartDefaultCommand()
            print(f"Placed chords at {chord_count} points")
        else:
            msg = "Requirements not met for construct chords"
            print(msg)
        return chord_count            

def get_tangent_at_point(detail: CurveLocationDetail)->RotMatrix:
    ''' Compute the tangent at a point on a line and return its orientation as a rotation matrix. '''
    vr: ValidatedDRay3d = detail.PointAndUnitTangent()
    ray: DRay3d = vr.Value()
    direction: DVec3d = ray.direction
    x_axis = DVec3d.From (1.0, 0.0, 0.0)
    angle = direction.AngleToXY(x_axis)
    _Z_AXIS = 2
    rotation = RotMatrix.FromAxisAndRotationAngle(_Z_AXIS, angle) 
    # Invert the rotation matrix so the cell is counter-rotated to match the line tangent
    rotation.Invert()
    return rotation  
    
if __name__ == "__main__":  # check if this script is being run directly (not imported as a module)
    msg = "Implementation of place cell along line"
    MessageCenter.ShowDebugMessage(msg, msg, False)

    cmdExecute(123, 3.3, 'a_cell')

